#' extractCNFeats
#'
#' This function is used to extract the six copy number features that are
#' eventually used in order to make the signatures. It does this using
#' six sub functions to extract each feature. Before extracting the features, 
#' the segments are passed through a validation function to make sure the data
#' is formatted correctly and there are no invalid segments.  Can be done in 
#' parallel using the cores parameter.
#'
#' @param segData The copy number segment data
#' @param gbuild The reference genome build. Default is hg19. Also supports hg18 and hg38.
#' @param cores The number of cores to use for parallel processing. Default 1.
#' @param featsToExtract The names of the features to extract.
#' @return list of dataframes containing results of six copy number features
#' @keywords features segsize bp10MB osCN bpchrarm changepoint
#' @import foreach
#' @export
#' @examples
#' extractCNFeats(segDataExp)
extractCNFeats = function(segData, gbuild = "hg19", cores = 1, featsToExtract = CNSigs::defaultFeats){

   #Get chromosome lengths and centromere positions from built in datasets
   if (gbuild == "hg19") {
     chrlen = hg19.chrom.sizes[1:24,]
     centromeres = gap_hg19[gap_hg19[,8] == "centromere",]
   } else if (gbuild == "hg18") {
     chrlen = hg18.chrom.sizes[1:24,]
     centromeres = gap_hg18[gap_hg18[,8] == "centromere",]
   } else if (gbuild == "hg38"){
     chrlen = hg38.chrom.sizes[1:24,]
     centromeres = gap_hg38[gap_hg38[,8] == "centromere",]
   } else {
     msg = "Unrecognized genome build. Check Parameters."
     stop(msg)
   }

   segData = validateSegData(segData, cores)
   
   
   numFeats = length(featsToExtract)
   #Caps cores at 1 for each feature
   cores = ifelse(cores > numFeats,numFeats,cores)

   #Creates cluseter of cores
   workers = parallel::makeCluster(cores, type = "SOCK")
   doParallel::registerDoParallel(workers)
   
   i=1
   #Assigns each feature to a different worker
   temp_list = foreach(i = 1:numFeats) %dopar% {
      featName = featsToExtract[i]
      switch(featName,
             "segsize" = list(segsize = extractSegsize(segData)),
             "bp10MB" = list(bp10MB = extractBP10MB(segData,chrlen)),
             "osCN" = list(osCN = extractOscillations(segData,chrlen)),
             "changepoint" = list(changepoint = extractChangepoints(segData)),
             "copynumber" = list(copynumber = extractCN(segData)),
             "bpchrarm" = list(bpchrarm = extractBPChrArm(segData, centromeres, chrlen)))
   }

   parallel::stopCluster(workers)
   unlist(temp_list, recursive = FALSE)
}

