#' fitModels
#'
#' This function takes all of the extracted copy number features and attempts to
#' fit a mixture of poisson and normal distributions to the data, and returns
#' a mixture of components that can be used to build the signatures. The order 
#' of features is "segsize","bp10MB","osCN","changepoint","copynumber","bpchrarm".
#' Therefore if you only want to change the maximum number of components for 
#' osCN to 5 then you would use max_comps = c(10,10,5,10,10,10).
#' @param CN_features List of features received from extractCopynumberFeatures
#' @param max_comps vector of length 6 specifying the max number of components 
#' for each feature. default is 10 for all features
#' @param min_comps vector of length 6 specifying the min number of components 
#' for each feature. default is 2 for all features
#' @param cores Number of parallel cores to use. Default is 1.
#' @param pR Peak Reduction reduces peaks in modeling to make modeling easier. Default is FALSE.
#' @param min_prior Used to override the minimum prior probabilty of a cluster
#' @param featsToModel The names of the features to extract.
#' @keywords mixed models components
#' @return Returns a list of the different components that contain flexmix 
#' objects for each feature
#' @import foreach
#' @export
#' @examples
#' \donttest{
#' fitModels(featsExp)}
#' 
#' #Models an exact number of components, useful when comparing two different
#' #datasets
#' min_comps = c(7, 3, 3, 2, 2, 3)
#' max_comps = c(7, 3, 3, 2, 10, 3)
#' fitModels(featsExp, max_comps, min_comps)
fitModels <- function(CN_features, max_comps = NULL, min_comps = NULL, cores = 1,
                      pR = FALSE, min_prior = NULL, featsToModel = CNSigs::defaultFeats) {
   numFeats = length(featsToModel)
   #Caps cores at 6
   cores = ifelse(cores > numFeats,numFeats,cores)
   
   if (!identical(names(CN_features),featsToModel)){
      stop("Given features do not match the featsToModel given.")
   }

   if (is.null(min_comps)){
      min_comps = rep(2,numFeats)
   }
   else if (length(min_comps) != numFeats){
      stop(paste("min_comps not of length ",numFeats))
   }

   if (is.null(max_comps)){
      max_comps = rep(10,numFeats)
   }
   else if (length(max_comps) != numFeats){
      stop("max_comps not of length ",numFeats)
   }
   
   #If you are looking for a specific number of components, reduce the
   # min_prior requirement to ensure the comps are found
   if (!is.null(min_prior))
      min_prior = min_prior
   else
      min_prior = 0.01

   

   workers = parallel::makeCluster(cores, type = "SOCK")
   doParallel::registerDoParallel(workers)
   
   i=1
   #Assigns each feature to a different worker
   CN_comps = foreach(i = 1:numFeats, .packages = c("flexmix")) %dopar% {
      toFit = CN_features[[i]][,2]
      featName = featsToModel[i]
      tryCatch({
      switch(featName,
             "segsize" = fitComponent(toFit,min_prior,min_comps[i],max_comps[i],pR=pR),
             "bp10MB" = fitComponent(toFit,min_prior,min_comps[i],max_comps[i],pR=pR),
             "osCN" = fitComponent(toFit,min_prior,min_comps[i],max_comps[i],"pois",pR),
             "changepoint" = fitComponent(toFit,min_prior,min_comps[i],max_comps[i],pR=pR),
             "copynumber" = fitComponent(toFit,min_prior,min_comps[i],max_comps[i],pR=pR),
             "bpchrarm" = fitComponent(toFit,min_prior,min_comps[i],max_comps[i],"pois",pR))
      },  error = function(err) {
         if (grepl("no convergence",err,fixed = TRUE)){
            err = paste0("Failed to converge while fitting ",names(CN_features)[i],
                        ", searched from ",min_comps[i]," to ",max_comps[i],
                        " components.")
         }
         stop(err)
      })
   }
   parallel::stopCluster(workers)
   
   #Rename and reorder to organize the components
   names(CN_comps) = featsToModel
   
   for (i in 1:length(CN_comps)){
      fittedComp = CN_comps[[i]]
      numComps = min(ncol(fittedComp),length(fittedComp))
      
      # If it is pois you have to order it differently since it only has one param
      if (is.null(ncol(fittedComp))){
         CN_comps[[i]] = fittedComp[order(fittedComp)]
         names(CN_comps[[i]]) = paste0("Comp.",1:numComps)
      } else {
         CN_comps[[i]] = fittedComp[,order(fittedComp[1,])]
         colnames(CN_comps[[i]]) = paste0("Comp.",1:numComps)
      }
   }
   
   return(CN_comps)
}
