#' plotSigExposureMat
#'
#' This function is used to generate the signature exposure matrix heatmap plot.
#' 
#' @param sigExposure Sample by signature matrix
#' @param runName Name of the run used in plot titles, Default is ""
#' @param saveRes Whether or not to save the plots, Default is FALSE
#' @param saveDir Where to save the plots, must be provided if using saveDir
#' @param rowOrder Ordering specification for the rows of the heatmap. Three possible options:
#' * TRUE:  Uses hierarchical clustering to determine row order.
#' * FALSE: (default) Leaves rows in the order they were given.
#' * A numeric vector the same length as the number of rows specifying the indices of the input matrix
#' @param colOrder Ordering specification for the columns of the heatmap. See above for options. Default value is T.
#' @return pheatmap figure of signature exposure by patient
#' @keywords plot sigExposure signature exposure matrix
#' @export
#' @examples
#' plotSigExposureMat(sigExposExp)
#' 
#' plotSigExposureMat(sigExposExp, rowOrder = FALSE, colOrder = FALSE)
#' 
#' newOrder = sample(1:ncol(sigExposExp), ncol(sigExposExp))
#' plotSigExposureMat(sigExposExp, colOrder = newOrder)
plotSigExposureMat = function(sigExposure, runName="", saveRes = FALSE, saveDir = NULL, rowOrder = FALSE, colOrder = TRUE) {
  
  #Checks the rowOrder parameter
  if (is.numeric(rowOrder)){
    if (length(rowOrder) != nrow(sigExposure)){
      msg = "rowOrder in plotSigExposureMat call does not have the correct amount of values."
      msg = paste0(msg,"\n Found ",length(rowOrder)," values, expected ",nrow(sigExposure),".")
      msg = paste0(msg,"\n Defaulting to heirarchical clustering for row order.")
      warning(msg)
      rowOrder = TRUE
    } else {
      sigExposure = sigExposure[rowOrder, ]
      rowOrder = FALSE
    }
  } else if (!is.logical(rowOrder)){
    msg = "rowOrder in plotSigExposureMat call in unrecognized format."
    msg = paste0(msg,"\n Please see documentation using ?plotSigExposureMat.")
    msg = paste0(msg,"\n Defaulting to heirarchical clustering for row order.")
    warning(msg)
    rowOrder = TRUE
  }
  
  #Checks the colOrder parameter
  if (is.numeric(colOrder)){
    if (length(colOrder) != ncol(sigExposure)){
      msg = "colOrder in plotSigExposureMat call does not have the correct amount of values."
      msg = paste0(msg,"\n Found ",length(colOrder)," values, expected ",ncol(sigExposure),".")
      msg = paste0(msg,"\n Defaulting to heirarchical clustering for column order.")
      warning(msg)
      colOrder = TRUE
    } else {
      sigExposure = sigExposure[,colOrder]
      colOrder = FALSE
    }
  } else if (!is.logical(colOrder)){
    msg = "colOrder in plotSigExposureMat call in unrecognized format."
    msg = paste0(msg,"\n Please see documentation using ?plotSigExposureMat.")
    msg = paste0(msg,"\n Defaulting to heirarchical clustering for column order.")
    warning(msg)
    colOrder = TRUE
  }
  
  # Generates the plot title based on the given runName
  if (nchar(runName) > 0){
    plotTitle = paste0("Patient by Signature matrix (",runName,")")
  } else {
    plotTitle = "Patient by Signature matrix"
  }
  
  # Generates the plot
  if (saveRes){
    if (is.null(saveDir)){
	stop('No saveDir provided for saveRes. Please set saveDir.')
    }
    grDevices::tiff(paste0(saveDir,"/PatientbySigMat",".tiff"),width = 8, height = 6,
                    units = 'in', res = 300)
    pheatmap::pheatmap(sigExposure, cluster_rows = rowOrder, cluster_cols = colOrder,
                       main = plotTitle, color = heatmapColors, border_color = NA,
                       treeheight_row = 0, treeheight_col = 0, show_colnames = FALSE)
    grDevices::dev.off()
  } else {
    pheatmap::pheatmap(sigExposure, cluster_rows = rowOrder, cluster_cols = colOrder,
                       main = plotTitle, color = heatmapColors, border_color = NA,
                       treeheight_row = 0, treeheight_col = 0, show_colnames = FALSE)
  }
}
