#' runPipeline
#'
#' This function allows you to run the entire Copy number signature pipeline
#' in one go. May take a while, especially if not given multiple cores. For 
#' more information on what actually happens in the pipeline, refer to the
#' CNSigs vignette.
#' 
#' @param segData The data to be analyzed. If a path name, readSegs is used to
#' make the list. Otherwise the list must be formatted correctly. Refer to 
#' ?readSegs for format information.
#' @param cores The number of computer cores to be used for parallel processing
#' @param nsigs The number of signatures to look for. Value of 0 runs the 
#' determineSigNum function to look for optimal number. Default is 0.
#' @param saveRes Whether or not to save the resulting tables and plots. 
#' Default is FALSE
#' @param runName Used to title plots and files when saving results
#' @param rmin Minimum number of signatures to look for. Default is 3.
#' @param rmax Maximum number of signatures to look for. Default is 12.
#' @param components Can be used when fixing components. Default is NULL.
#' @param max_comps vector of length 6 specifying the max number of components 
#' for each feature. Passed to fitModels. Default is 10 for all features
#' @param min_comps vector of length 6 specifying the min number of components
#'  for each feature. Passed to fitModels. Default is 2 for all features
#' @param fixedSigs Signature x Component matrix. Used when fixing signatures.
#'  Default is NULL
#' @param saveDir Used to specify where to save the results, must be provided 
#' if using saveDir
#' @param smooth Whether or not to smooth the input data. Default is F.
#' @param colMap Mapping of column names when reading from text file. 
#' Default column names are ID, chromosome, start, end, segVal.
#' @param pR Peak Reduction
#' @param gbuild The reference genome build. Default is hg19. Also supports hg18 and hg38.
#' @param featsToUse Vector of feature names that you wish to use
#' @param ploidyData The ploidy data to use as a feature
#' @param plot Whether or not to generate the plots. Default is T. 
#' @keywords run
#' @return  Returns a list with all of the results from the pipeline
#' @export
#' @examples
#' #Runs the entire pipeline on the example data giving it 6 cores and specifying
#' #5 signatures with a name of "TCGA Test"
#' \donttest{
#' runPipeline(segDataExp, cores = 6, nsigs = 5, saveRes = FALSE, "TCGA Test")}
#' 

runPipeline <- function(segData, cores = 1, nsigs = 0, saveRes = FALSE, 
                        runName = "Run", rmin = 3, rmax = 12,
                        components = NULL, max_comps = NULL, 
                        min_comps = NULL, fixedSigs = NULL, saveDir = NULL,
                        smooth = FALSE, colMap = NULL,pR=FALSE,gbuild = "hg19",
                        featsToUse = NULL, ploidyData = NULL, plot = TRUE){
   
   # Creates empty list to hold results 
   results = list()
   results[["func"]] = sys.call()

   #Enforces default features if none were specified
   if (is.null(featsToUse)){
      featsToUse = CNSigs::defaultFeats;
   }
   
   # Checks the ploidy related parameters to see if it needs to read the data in
   readPloidy = F
   if ("ploidy" %in% featsToUse){
        readPloidy = is.null(ploidyData)
        featsToUse = featsToUse[featsToUse != "ploidy"]
   } 

   # Checks if ploidy is in fixed sigs and not given, If so throw an error.
   if (!is.null(fixedSigs)){
	if ("log2Ploidy" %in% rownames(fixedSigs) && is.null(ploidyData)){
		stop("The pipeline detected that your fixed signatures expect ploidy data and has no input ploidy data to use. Please either provide signatures without ploidy data or provide ploidyData in the runPipeline function.")
	}
   }

   # Sets up the segData either using readSegs or making sure it is a list that
   # is formatted correctly
   if (is.character(segData)){
      segData = readSegs(segData, colMap, readPloidy)
   }
   else if (!is.list(segData)){
      stop("Input segData must be in a list format. Refer to ?readSegs for specifics on data format")
   }
   else if (!identical(names(segData[[1]]),c("chromosome","start","end","segVal"))){
      msg = paste0("Input segData column names not correct.\n  ",
                   "Expected: chromosome, start, end, segVal\n  ",
                   "Got: ", paste(names(segData[[1]]),collapse = ", "),"\n  ",
                   "Refer to ?readSegs for specifics on data format")
      stop(msg)
   }
   
   if (readPloidy){
      ploidyData = segData[[2]]
      segData = segData[[1]]
   }
   
   message("Using ",length(segData)," samples for the analysis.\n")
   
   # Smooths the data if specified
   if (smooth){
      results[["Raw Input"]] = segData
      message("Smoothing Segments.\n")
      segData = smoothSegs(segData, cores)
      results[["Input_data"]] = segData
   }
   else {
      results[["Input_data"]] = segData
   }
   
   # Constructs the full path for the save directory
   if (is.null(saveDir)){
      saveDir = paste0(getwd(),"/CNSig Results");
   }
   else {
      saveDir = paste0(saveDir,"/CNSig Results");
   }

   #When saveRes=TRUE it makes a new folder with runName and the date to save to
   if (saveRes){
      if (is.null(saveDir)){
	stop('No saveDir provided for saveRes. Please set saveDir.')
      }
      if (!dir.exists(saveDir)){
         message("Making new results directory at ",saveDir,"\n")
         dir.create(saveDir)
      }

      #Finds runs performed today
      todayResults = list.files(saveDir,pattern=paste0("^",Sys.Date(),"_",runName))

      if (length(todayResults) == 0){
         saveDir = paste0(saveDir,"/",Sys.Date(),"_",runName)
      }
      else {
         saveDir = paste0(saveDir,"/",Sys.Date(),"_",runName,"_",length(todayResults))
      }
      dir.create(saveDir)
      message("Saving results in ",saveDir,"\n")
   }
   
   message("Extracting Copy Number Features.\n")
   results[["CN_features"]] = extractCNFeats(segData,gbuild,cores,featsToUse)

   #If the component object is null, then perform the modeling
   if (is.null(components)){
      message("Fitting the component models.\n")
      results[["CN_components"]] = fitModels(results[["CN_features"]],
                                             max_comps, min_comps,cores,pR,featsToModel=featsToUse)
   }
   else {
      results[["CN_components"]] = components
   }

   # Creates the scm
   message("Generating sample-component-matrix.\n")
   results[["scm"]] = generateSCM(results[["CN_features"]],results[["CN_components"]])
   
   
   if (!is.null(ploidyData)){
      results[["scm"]] = addPloidyData(results[["scm"]], ploidyData)
   }
   
   # If the number of signatures is not specified determine the number of signatures
   # and prompt the user to pick a number.
   if (nsigs == 0 && is.null(fixedSigs)){
      message("Determining # of Signatures via NMF.\n")
      determineNumSigs(results[["scm"]], rmin, rmax,cores,saveRes=saveRes,saveDir = saveDir)
      input = readline("Input number of signatures to use for analysis: ")
      nsigs = as.numeric(input)
   }

   # IF a set of signatures is not given new signatures are found, otherwise it
   # finds the exposures to the given signatures
   if (is.null(fixedSigs)){
      message("Generating Signatures.\n")
      results[["nmf_Results"]] = createSigs(results[["scm"]],nsigs,cores,runName,saveRes,saveDir)
      results[["sigExposure"]] = as.data.frame(scoef(results[["nmf_Results"]]))
      results[["sigs"]] = NMF::basis(results[["nmf_Results"]])
   }
   else {
      message("Finding Signature Exposures.\n")
      results[["sigs"]]=fixedSigs
      results[["sigExposure"]] = findExposures(t(results[["scm"]]),fixedSigs,runName,saveRes,saveDir)
      
      results[["nmf_Results"]] = NULL
   }
   
   
   if (plot) {
      # Plots the scm
      plotScm(results[["scm"]],runName,saveRes,saveDir)
      
      # Plot the signature matrix and exposure matrix
      plotSigMat(results[["sigs"]],runName,saveRes,saveDir)
      plotSigExposureMat(results[["sigExposure"]],runName,saveRes,saveDir)
      
      #Only plot the signatures if de-novo
      if (is.null(fixedSigs)){
         # Plots the signatures
         message("Plotting Signatures.\n")
         plotSigs(results[["sigs"]],saveRes,saveDir,runName)
      }
   }
   
   if (saveRes){
      if (is.null(saveDir)){
	stop('No saveDir provided for saveRes. Please set saveDir.')
      }
      message("Saving Pipeline Results.\n")
      saveRDS(results,file=paste0(saveDir,'/Pipeline Results.rds'))
   }
   
   return(results)
}
