% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/D3mirt.R
\name{D3mirt}
\alias{D3mirt}
\title{3D DMIRT Model Estimation}
\usage{
D3mirt(
  x,
  modid = NULL,
  model = NULL,
  con.items = NULL,
  con.sphe = NULL,
  itemtype = "graded",
  method.mirt = "QMCEM",
  method.fscores = "EAP",
  QMC = TRUE
)
}
\arguments{
\item{x}{A data frame with items in rows and model parameters in columns containing raw response data as integer values or factor loadings.
Input can also be an S4 object of class 'SingleGroupClass' exported from \link[mirt:mirt]{mirt::mirt} (Chalmers, 2012).
Regarding the data frame, the number of columns must be more than or equal to 4, i.e., three columns with (\emph{a}) parameters and at least one column for difficulty (\emph{d}) parameters.}

\item{modid}{Use either the two model identification items from \link{modid} as a combined vector or use nested list of item indicators to fit an orthogonal model (see examples below).
The default is \code{modid = NULL}.}

\item{model}{The user has the option of imputing a model specification schema used in the call to \link[mirt:mirt]{mirt::mirt} (Chalmers, 2012).
The default is \code{model = NULL}.}

\item{con.items}{Optional. Nested lists with integer values as item indicators to identify constructs. The default is \code{con.items = NULL}.}

\item{con.sphe}{Optional. Nested lists of spherical angles to identify constructs. The default is \code{con.sphe = NULL}.}

\item{itemtype}{What item type to use in the function call. Available options are \code{"2PL"} and \code{"graded"}. The default is \code{itemtype = "graded"}.}

\item{method.mirt}{Estimation algorithm for \link[mirt:mirt]{mirt::mirt} (Chalmers, 2012) to fit the model. The default is \code{method.mirt = "QMCEM"}.}

\item{method.fscores}{Factor estimation algorithm for \link[mirt:fscores]{mirt::fscores} (Chalmers, 2012) for extracting respondent trait scores. The default is \code{method.fscores = "EAP"}.}

\item{QMC}{Integration method for \link[mirt:fscores]{mirt::fscores} (Chalmers, 2012). The default is \code{QMC = TRUE}.}
}
\value{
A S3 object of class \code{D3mirt} with lists of \emph{a} and \emph{d} parameters from the M2PL or MGRM estimation, multidimensional difficulty (MDIFF), multidimensional discrimination (MDISC), direction cosines and degrees for vector angles, construct lists, vector coordinates, and respondent trait scores.
}
\description{
Descriptive multidimensional item response theory model estimation (DMIRT; Reckase, 2009, 1985, Reckase and McKinley, 1991) for dichotomous and polytomous items restricted to three dimensions.
}
\details{
The \code{D3mirt()} function takes in model parameters from a compensatory three-dimensional multidimensional two-parameter logistic model (M2PL) or a multidimensional graded
response model (MGRM), either in the form of a data frame with item data, or a data frame with factor loadings or an S4 object of class 'SingleGroupClass' exported from \link[mirt:mirt]{mirt::mirt} (Chalmers, 2012) function fitted in accordance with descriptive item response theory model specifications (see package vignette).
The function returns DMIRT estimates that can be visualized with \link{plot} that graph vector arrows representing item response characteristics in a three-dimensional space.
Regarding the former, this includes visualization of the single multidimensional discrimination (MDISC) parameter and the multidimensional difficulty (MDIFF) parameters (Reckase, 2009, 1985; Reckase & McKinley, 1991).
The function also returns respondent trait scores that can be plotted with \link{plot} as spheres located in the three-dimensional latent space.
In turn, this allows for studying respondent profiles using the plot function (for more on profiles, see function documentation on \link{plot}).

There are two types of models available for D3mirt analysis. The default model is the basic DMIRT model (Reckase, 2009, 1985, Reckase & McKinley, 1991) that relaxes the assumption of unidimensionality in the items while restricting the latent space to be orthogonal.
To use the default option requires first selecting two items to identify the model. This can be done manually with the \code{modid} argument in the function call to \code{D3mirt}.
However, it is advisable to use the dedicated function \link{modid} included in the package for this purpose (for more on model identification see function documentation for \link{modid}).
In contrast, the optional orthogonal model constrains the items to be strictly parallel with the axes (see example section below).
Consequently, this option allows the user to investigate the model under the assumption that the items are strictly unidimensional and orthogonally oriented in the latent space.
In this context "orthogonal" refers to the perpendicular orientation of the item vectors the model specification creates.
Note that using the optional model will also affect respondent locations in the latent space accordingly.
It is also possible to specify a unique model with the help of the \code{model} argument in the function call to \code{D3mirt} if written in mirt (Chalmers, 2012) syntax (for an example, see the appendix in the package vignette).

The user also has the option of including constructs in the estimation.
Constructs, in this context, refer to the assumption that a subset of items or a particular angle in the latent space holds some higher-order latent variable of interest.
Constructs are visualized when plotting as solid black arrows running across the model space.
In addition, if constructs are used, the output will also contain the directional discrimination (DDISC) parameters for all items assessed in the direction indicated by the construct vectors.
This makes it possible to compare item discrimination under the assumption that the items are unidimensional, measuring the same latent variable indicated by the angle of the construct vector.

To include constructs, the user can create one or more nested lists that indicate what items belong to what construct (from one item to all items in the set; see the examples section below).
From this, the \code{D3mirt()} function calculates the average direction by adding and normalizing the direction cosines using the items in the nested lists.
Constructs can also be indicated using spherical coordinates stored in nested lists.
This allows the user to freely add any number of constructs at any angle in the latent space to study the item discrimination.

For more on theory and how to interpret statistical estimates, please see the package vignette.
}
\examples{
\donttest{
# Load data
data("anes0809offwaves")
x <- anes0809offwaves
x <- x[, 3:22] # Remove columns for age and gender

# Call to D3mirt(), including optional nested lists for three constructs
# Item W7Q16 is not included in any construct because of model violations
# Constructs can also be defined using interval notation, i.e., c(1:10) and so on.

con <- list(c(1,2,3,4,5,6,7,8,9,10),
            c(11,12,13,14),
            c(15,17,18,19,20))
mod <- D3mirt(x, modid = c("W7Q3", "W7Q20"), con.items = con)

# Show summary of results
summary(mod)

# Call to D3mirt(), including optional constructs with the help of spherical coordinates
# Spherical coordinates are indicated using nested list structures with angles
# First angle indicates the rotation in the xz-plane
# The second angle is the angle away from the y-axis.
# The specification below indicates three constructs located at a 45-degree angle
# between the three axes in the positive orientation.
# It is possible to assign factor loadings and difficulty parameters from mod to a new data frame
# This skips fitting the compensatory model and makes fitting the model with D3mirt() instant
# Note that trait scores will not be included in the exported S3 object when using this option
y <- cbind(mod$loadings, mod$diff)
con <- list(c(0, 45),
            c(45, 90),
            c(90, 45))
mod <- D3mirt(y, con.sphe = con)

# Show summary of results
summary(mod)

# Call D3mirt() using the orthogonal optional model
# often requires removing items with poor fit
# In this example item W7Q16 is removed from the data frame
y <- data.frame(x[,-16])

# Items are constrained to the x, y, and z-axes using
# nested lists with positive integers as item indicators
# Note that integers indicate where the items are located in the data frame
mod <- D3mirt(y, modid = list(c(1:10),
                            c(15:19),
                            c(11:14)))

# Show summary of results
summary(mod)
}
}
\references{
Chalmers, R., P. (2012). mirt: A Multidimensional Item Response Theory Package for the R Environment. \emph{Journal of Statistical Software, 48}(6), 1-29.\cr
https://doi.org/10.18637/jss.v048.i06

Reckase, M. D. (2009). \emph{Multidimensional Item Response Theory}. Springer.

Reckase, M. D. (1985). The Difficulty of Test Items That Measure More Than One Ability. \emph{Applied Psychological Measurement, 9}(4), 401-412.\cr
https://doi.org/10.1177/014662168500900409

Reckase, M. D., & McKinley, R. L. (1991). The Discriminating Power of Items That Measure More Than One Dimension. \emph{Applied Psychological Measurement, 15}(4), 361-373.\cr
https://doi.org/10.1177/014662169101500407
}
\author{
Erik Forsberg
}
