% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ConditionalTernary.R
\name{conditional_ternary}
\alias{conditional_ternary}
\title{DI specific wrapper for conditional ternary diagrams}
\usage{
conditional_ternary(
  model,
  FG = NULL,
  values = NULL,
  tern_vars = NULL,
  conditional = NULL,
  add_var = list(),
  resolution = 3,
  plot = TRUE,
  nlevels = 7,
  colours = NULL,
  lower_lim = NULL,
  upper_lim = NULL,
  contour_text = FALSE,
  show_axis_labels = TRUE,
  show_axis_guides = FALSE,
  axis_label_size = 4,
  vertex_label_size = 5,
  nrow = 0,
  ncol = 0
)
}
\arguments{
\item{model}{A Diversity Interactions model object fit by using the
\code{\link[DImodels:DI]{DI()}} function from the
\code{\link[DImodels:DImodels-package]{DImodels}} package.}

\item{FG}{A character vector specifying the grouping of the variables
specified in `prop`. Specifying this parameter would call the
grouped_ternary_data function internally. See \code{\link{grouped_ternary}}
or \code{\link{grouped_ternary_data}} for more information.}

\item{values}{A numeric vector specifying the proportional split of the
variables within a group. The default is to split the group
proportion equally between each variable in the group.}

\item{tern_vars}{A character vector giving the names of the three variables
to be shown in the ternary diagram.}

\item{conditional}{A data-frame describing the names of the compositional variables
and their respective values at which to slice the
simplex space. The format should be, for example, as follows: \cr
\code{data.frame("p1" = c(0, 0.5), "p2" = c(0.2, 0.1))} \cr
One figure would be created for each row in `conditional` with
the respective values of all specified variables. Any
compositional variables not specified in `conditional` will
be assumed to be 0.}

\item{add_var}{A list or data-frame specifying values for additional variables
in the model other than the proportions (i.e. not part of the
simplex design).
This could be useful for comparing the predictions across
different values for a non-compositional variable.
If specified as a list, it will be expanded to show a plot
for each unique combination of values specified, while if specified
as a data-frame, one plot would be generated for each row in the
data.}

\item{resolution}{A number between 1 and 10 describing the resolution of the
resultant graph.
A high value would result in a higher definition figure
but at the cost of being computationally expensive.}

\item{plot}{A boolean variable indicating whether to create the plot or return
the prepared data instead. The default \code{TRUE} creates the plot while
\code{FALSE} would return the prepared data for plotting. Could be useful
if user wants to modify the data first and then create the plot.}

\item{nlevels}{The number of levels to show on the contour map.}

\item{colours}{A character vector or function specifying the colours for the
contour map or points. The number of colours should be same as
`nlevels` if (`show = "contours"`). \cr
The default colours scheme is the
\code{\link[grDevices:terrain.colors]{terrain.colors()}} for
continuous variables and an extended version of the Okabe-Ito
colour scale for categorical variables.}

\item{lower_lim}{A number to set a custom lower limit for the contour
(if `show = "contours"`). The default is minimum of the prediction.}

\item{upper_lim}{A number to set a custom upper limit for the contour
(if `show = "contours"`). The default is maximum of the prediction.}

\item{contour_text}{A boolean value indicating whether to include labels on
the contour lines showing their values
(if `show = "contours"`). The default is \code{FALSE}.}

\item{show_axis_labels}{A boolean value indicating whether to show axis
labels along the edges of the ternary. The default
is \code{TRUE}.}

\item{show_axis_guides}{A boolean value indicating whether to show axis
guides within the interior of the ternary. The
default is \code{FALSE}.}

\item{axis_label_size}{A numeric value to adjust the size of the axis labels
in the ternary plot. The default size is 4.}

\item{vertex_label_size}{A numeric value to adjust the size of the vertex
labels in the ternary plot. The default size is 5.}

\item{nrow}{Number of rows in which to arrange the final plot
(when `add_var` is specified).}

\item{ncol}{Number of columns in which to arrange the final plot
(when `add_var` is specified).}
}
\value{
A ggmultiplot (ggplot if single plot is returned) class object or data-frame (if `plot = FALSE`)
}
\description{
We fix \eqn{n-3} variables to have a constant value \eqn{p_1, p_2, p_3, ... p_{n-3}}
such that \eqn{P = p_1 + p_2 + p_3 + ... p_{n - 3}} and \eqn{0 \le P \le 1} and
vary the proportion of the remaining three variables between \eqn{0} and \eqn{1-P}
to visualise the change in the predicted response as a contour map within a
ternary diagram. This is equivalent to taking multiple 2-d slices of the
high dimensional simplex space. Taking multiple 2-d slices across multiple
variables should allow to create an approximation of how the response varies
across the n-dimensional simplex.
This is a wrapper function specifically for statistical models fit using the
\code{\link[DImodels:DI]{DI()}} function from the
\code{\link[DImodels:DImodels-package]{DImodels}} R package and would implicitly
call \code{\link{conditional_ternary_data}} followed by
\code{\link{conditional_ternary_plot}}. If your model object isn't fit using
DImodels, consider calling these functions manually.
}
\examples{
library(DImodels)
library(dplyr)
data(sim2)
m1 <- DI(y = "response", data = sim2, prop = 3:6, DImodel = "FULL")

#' ## Create data for slicing
## We only condition on the variable "p3"
conditional_ternary(model = m1, tern_vars = c("p1", "p2", "p4"),
                    conditional = data.frame("p3" = c(0, 0.2, 0.5)),
                    resolution = 1)

## Slices for experiments for over 4 variables
data(sim4)
m2 <- DI(y = "response", prop = paste0("p", 1:6),
         DImodel = "AV", data = sim4) \%>\%
         suppressWarnings()

## Conditioning on multiple variables
cond <- data.frame(p4 = c(0, 0.2), p3 = c(0.5, 0.1), p6 = c(0, 0.3))
conditional_ternary(model = m2, conditional = cond,
                    tern_vars = c("p1", "p2", "p5"), resolution = 1)

## Create separate plots for additional variables not a part of the simplex
m3 <- DI(y = "response", prop = paste0("p", 1:6),
         DImodel = "AV", data = sim4, treat = "treatment") \%>\%
         suppressWarnings()

## Create plot and arrange it using nrow and ncol
\donttest{
conditional_ternary(model = m3, conditional = cond[1, ],
                    tern_vars = c("p1", "p2", "p5"),
                    resolution = 1,
                    add_var = list("treatment" = c(50, 150)),
                    nrow = 2, ncol = 1)
}

## Specify `plot = FALSE` to not create the plot but return the prepared data
head(conditional_ternary(model = m3, conditional = cond[1, ],
                         resolution = 1, plot = FALSE,
                         tern_vars = c("p1", "p2", "p5"),
                         add_var = list("treatment" = c(50, 150))))
}
