% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/easyEGO.R
\name{easyEGO}
\alias{easyEGO}
\title{User-friendly wrapper of the functions \code{\link[DiceOptim]{fastEGO.nsteps}} and \code{\link[DiceOptim]{TREGO.nsteps}}.
Generates initial DOEs and kriging models (objects of class \code{\link[DiceKriging]{km}}), 
and executes \code{nsteps} iterations of either EGO or TREGO.}
\usage{
easyEGO(
  fun,
  budget,
  lower,
  upper,
  X = NULL,
  y = NULL,
  control = list(trace = 1, seed = 42),
  n.cores = 1,
  ...
)
}
\arguments{
\item{fun}{scalar function to be minimized,}

\item{budget}{total number of calls to the objective and constraint functions,}

\item{lower}{vector of lower bounds for the variables to be optimized over,}

\item{upper}{vector of upper bounds for the variables to be optimized over,}

\item{X}{initial design of experiments. If not provided, X is taken as a maximin LHD with budget/3 points}

\item{y}{initial set of objective observations \eqn{f(X)}. Computed if not provided.}

\item{control}{an optional list of control parameters. See "Details".}

\item{n.cores}{number of cores for parallel computation}

\item{...}{additional parameters to be given to \code{fun}}
}
\value{
A list with components:
\itemize{
\item \code{par}: the best feasible point
\item \code{values}: a vector of the objective and constraints at the point given in \code{par},
\item \code{history}: a list containing all the points visited by the algorithm (\code{X}) and their corresponding objectives (\code{y}).
\item \code{model}: the last GP model, class \code{\link[DiceKriging]{km}}
\item \code{control}: full list of control values, see "Details"
\item \code{res}: the output of either \code{\link[DiceOptim]{fastEGO.nsteps}} or \code{\link[DiceOptim]{TREGO.nsteps}}
}
}
\description{
User-friendly wrapper of the functions \code{\link[DiceOptim]{fastEGO.nsteps}} and \code{\link[DiceOptim]{TREGO.nsteps}}.
Generates initial DOEs and kriging models (objects of class \code{\link[DiceKriging]{km}}), 
and executes \code{nsteps} iterations of either EGO or TREGO.
}
\details{
Does not require knowledge on kriging models (objects of class \code{\link[DiceKriging]{km}})  \cr
The \code{control} argument is a list that can supply any of the following components: \cr
\itemize{
\item \code{trace}: between -1 and 3
\item \code{seed}: to fix the seed of the run
\item \code{cov.reestim}: Boolean, if TRUE (default) the covariance parameters are re-estimated at each iteration
\item \code{model.trend}: trend for the GP model
\item \code{lb, ub}: lower and upper bounds for the GP covariance ranges
\item \code{nugget}: optional nugget effect
\item \code{covtype}: covariance of the GP model (default "matern5_2")
\item \code{optim.method}: optimisation of the GP hyperparameters (default "BFGS")
\item \code{multistart}: number of restarts of BFGS
\item \code{gpmean.trick, gpmean.freq}: Boolean and integer, resp., for the gpmean trick
\item \code{scaling}: Boolean, activates input scaling
\item \code{warping}: Boolean, activates output warping
\item \code{TR}: Boolean, activates TREGO instead of EGO
\item \code{trcontrol}: list of parameters of the trust region, see \code{\link[DiceOptim]{TREGO.nsteps}}
\item \code{always.sample}: Boolean, activates force observation even if it leads to poor conditioning
}
}
\examples{
library(parallel)
library(DiceOptim)
set.seed(123)

#########################################################
### 	10 ITERATIONS OF TREGO ON THE BRANIN FUNCTION, ####
###	 STARTING FROM A 9-POINTS FACTORIAL DESIGN       ####
########################################################

# a 9-points factorial design, and the corresponding response
ylim=NULL
fun <- branin; d <- 2
budget <- 5*d
lower <- rep(0,d)
upper <- rep(1,d)
n.init <- 2*d

control <- list(n.init=2*d, TR=TRUE, nugget=1e-5, trcontrol=list(algo="TREGO"), multistart=1)

res1 <- easyEGO(fun=fun, budget=budget, lower=lower, upper=upper, control=control, n.cores=1)

par(mfrow=c(3,1))
y <- res1$history$y
steps <- res1$res$all.steps
success <- res1$res$all.success
sigma <- res1$res$all.sigma
ymin <- cummin(y)
pch <- rep(1, length(sigma))
col <- rep("red", length(sigma))
pch[which(!steps)] <- 2
col[which(success)] <- "darkgreen"

pch2 <- c(rep(3, n.init), pch)
col2 <- c(rep("black", n.init), col)
plot(y, col=col2, ylim=ylim, pch=pch2, lwd=2, xlim=c(0, budget))
lines(ymin, col="darkgreen")
abline(v=n.init+.5)

plot(n.init + (1:length(sigma)), sigma, xlim=c(0, budget), ylim=c(0, max(sigma)), 
pch=pch, col=col, lwd=2, main="TR size") 
lines(n.init + (1:length(sigma)), sigma, xlim=c(0, budget)) 
abline(v=n.init+.5)

plot(NA, xlim=c(0, budget), ylim=c(0, 1), main="x0 (coordinates)")
for (i in 1:d) {
  lines(n.init + (1:nrow(res1$res$all.x0)), res1$res$all.x0[,i]) 
  points(n.init + (1:nrow(res1$res$all.x0)), res1$res$all.x0[,i], pch=pch, col=col, lwd=2) 
}
abline(v=n.init+.5)

par(mfrow=c(1,1))
pairs(res1$model@X, pch=pch2, col=col2)

}
\references{
D.R. Jones, M. Schonlau, and W.J. Welch (1998), Efficient global
optimization of expensive black-box functions, \emph{Journal of Global
Optimization}, 13, 455-492.
}
\author{
Victor Picheny
}
