% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ClustF.R
\name{ClustF}
\alias{ClustF}
\title{Clustering Coefficients for Directed/Undirected and Weighted Networks}
\usage{
ClustF(mat, type = "undirected", isolates = "zero", norm = 1)
}
\arguments{
\item{mat}{A weighted adjacency matrix. If weights are greater than one,
a normalization is provided by dividing each weight by the maximum weight observed.}

\item{type}{The type of clustering coefficient to calculate.
Possible values are: \code{"undirected"} (default) or \code{"directed"}.}

\item{isolates}{Character scalar, defines how to treat vertices with degree zero and one.
If \code{"NaN"}, their local transitivity is reported as NaN and they are not included in the averaging.
If \code{"zero"}, their transitivity is reported as 0 and they are included in the averaging. Default is \code{"zero"}.}

\item{norm}{If it is 1 (default), the link's weights are normalized by dividing by the maximum observed weight (as proposed by Fagiolo).
If it is 0, weights are not normalized. Weights are always normalized when the maximum weight is greater than zero, ensuring that the clustering coefficient ranges between 0 and 1.}
}
\value{
A list with the following components:
\item{LocalCC}{Local clustering coefficients for undirected networks}
\item{GlobalCC}{Global clustering coefficient for undirected networks}
\item{cycleCC}{Local Cycle clustering coefficients for directed networks}
\item{middlemanCC}{Local Middleman clustering coefficients for directed networks}
\item{inCC}{Local In clustering coefficients for directed networks}
\item{outCC}{Local Out clustering coefficients for directed networks}
\item{totalCC}{Local Total clustering coefficients for directed networks}
\item{GlobalcycleCC}{Global Cycle clustering coefficient for directed networks}
\item{GlobalmiddlemanCC}{Global Middleman clustering coefficient for directed networks}
\item{GlobalinCC}{Global In clustering coefficient for directed networks}
\item{GlobaloutCC}{Global Out clustering coefficient for directed networks}
\item{GlobaltotalCC}{Global Total clustering coefficient for directed networks}
}
\description{
This function computes both Local and Global (average) Clustering Coefficients for either Directed/Undirected and Unweighted/Weighted Networks.
The formulas are based on Onnela et al. (2005) for undirected networks,
and on Fagiolo (2007) for directed networks.
}
\details{
In the directed case, different components of the directed clustering coefficient are also considered.

The function computes Onnela et al.'s (2005) formula for weighted and undirected networks.
For directed networks, Fagiolo's (2007) formula is used.
In the case of unweighted and undirected graphs, the classical local clustering coefficient (Watts and Strogatz) is provided.
Local coefficients are computed for each node, and the global coefficient is the average of these local coefficients.
These coefficients do not work for graphs with multiple or loop edges, so loops are removed.
}
\examples{
if (requireNamespace("igraph", quietly = TRUE)) {
  library(igraph)
  # Generate a weighted and undirected graph
  gsim <- sample_gnp(50, 0.5, directed = FALSE, loops = FALSE)
  PESI <- runif(length(E(gsim)), 0, 1)
  E(gsim)$weight <- PESI
  A <- as_adjacency_matrix(gsim, sparse = FALSE, attr = "weight")
  OnnelaClust <- ClustF(A, "undirected")

  # Generate a weighted and directed graph
  gsim <- sample_gnp(50, 0.5, directed = TRUE, loops = FALSE)
  PESI <- runif(length(E(gsim)), 0, 1)
  E(gsim)$weight <- PESI
  A <- as_adjacency_matrix(gsim, sparse = FALSE, attr = "weight")
  FagioloClust <- ClustF(A, "directed")
} else {
  cat("Please install the 'igraph' package to run this example.\n")
}

}
\references{
Fagiolo, G. (2007). Clustering in complex directed networks. \emph{Physical Review E}, 76(2).

Onnela, J.P., Saramaki, J., Kertsz, J., & Kaski, K. (2005). Intensity and coherence of motifs in weighted complex networks. \emph{Physical Review E}, 71(6).

Watts, D.J., & Strogatz, S.H. (1998). Collective dynamics of 'small-world' networks. \emph{Nature}, 393, 440-442.
}
\author{
Gian Paolo Clemente, \email{gianpaolo.clemente@unicatt.it}
}
