% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EquiTrends_Main_Functions.R
\name{rmsEquivTest}
\alias{rmsEquivTest}
\title{Equivalence Test for Pre-trends based on the RMS Placebo Coefficient}
\usage{
rmsEquivTest(
  Y,
  ID,
  G,
  period,
  X = NULL,
  data = NULL,
  equiv_threshold = NULL,
  pretreatment_period = NULL,
  base_period = NULL,
  alpha = 0.05,
  no_lambda = 5
)
}
\arguments{
\item{Y}{A numeric vector with the variable of interest. If \code{data} is supplied, \code{Y} should be a scalar indicating the column number or column-name character string that corresponds to the numeric dependent (outcome) variable in ’data’.}

\item{ID}{A numeric vector identifying the different cross-sectional units in the dataset. If \code{data} is supplied, \code{ID} should be a scalar indicating the column number or column-name character string that corresponds to the cross-sectional units identifier in \code{data}.}

\item{G}{A binary or logic vector (of the same dimension as \code{Y} and \code{ID}) indicating if the individual (e.g. as indicated by \code{ID}) receives treatment (e.g. 1 or TRUE) or not (0 or FALSE). f 'data' is supplied, \code{G} should be a scalar identifying the column number or column-name character string associated to \code{G} in \code{data}.}

\item{period}{A numeric vector (of the same dimension as Y) indicating time. If \code{data} is supplied, \code{period} should be a scalar indicating the column number or column-name character string that corresponds to the time identifier in \code{data}.}

\item{X}{A vector, matrix, or data.frame containing the control variables. If \code{data} is supplied, \code{X} must be a vector of column numbers or column-name character strings that identifies the control variables’ columns.}

\item{data}{An optional \code{data.frame} object containing the variables in Y, ID, G, T and, if supplied, X and cluster as its columns.}

\item{equiv_threshold}{The scalar equivalence threshold (must be positive). The default is NULL, implying that the function must look for the minimum value for which the null hypothesis of ”non-negligible differences” can still be rejected.}

\item{pretreatment_period}{A numeric vector identifying the pre-treatment periods that should be used for testing. \code{pretreatment_period} must be a subset of the periods included through \code{period}. The default is to use all periods that are included in \code{period}.}

\item{base_period}{The pre-treatment period to compare the post-treatment observation to. The default is to take the last period of the pre-treatment period.}

\item{alpha}{Significance level of the test. The default is 0.05.}

\item{no_lambda}{Parameter specifying the number of incremental segments of the dataset over which a statistic is calculated. See \emph{Details}. The default is 5.}
}
\value{
An object of class "rmsEquivTest" containing:
\item{\code{placebo_coefficients}}{A numeric vector of the estimated placebo coefficients,}
\item{\code{rms_placebo_coefs}}{the root mean squared value of the placebo coefficients,}
\item{\code{significance_level}}{the significance level of the test,}
\item{\code{base_period}}{the base period used in the testing procedure,}
\item{\code{num_individuals}}{the number of cross-sectional individuals in the panel used for testing,}
\item{\code{num_periods}}{the number of pre-treatment periods in the panel used for testing (if the panel is unbalanced, \code{num_periods} represents the range in the number of time periods covered by different individuals),}
\item{\code{num_observations}}{the total number of observations in the panel used for testing,}
\item{\code{is_panel_balanced}}{a logical value indicating whether the panel is balanced,}
\item{\code{equiv_threshold_specified}}{a logical value indicating whether an equivalence threshold was specified.}

If \code{equiv_threshold_specified = FALSE}, then additionally \code{minimum_equiv_threshold}: the minimum equivalence threshold for which the null hypothesis of non-negligible (based on the equivalence threshold) trend-differences can be rejected. 

If \code{equiv_threshold_specified = TRUE}, then additionally
\itemize{
\item \code{rms_critical_value}: the critical value at the alpha level,
\item \code{reject_null_hypothesis}: A logical value indicating whether to reject the null hypothesis,
\item \code{equiv_threshold}: the equivalence threshold specified.
}
}
\description{
This function performs an equivalence test for pre-trends based on the root mean squared placebo coefficient from Dette & Schumann (2024).
}
\details{
\code{no_lambda} determines the proportions lambda/\code{no.lambda} for lambda = 1,...,\code{no_lambda} of the cross-sectional units at which the placebo coefficients are estimated. The placebo coefficients are estimated for each of these proportions and the root mean squared (RMS) of the placebo coefficients is calculated, which are then used to construct the critical value at a significance level of \code{alpha}. See Dette & Schumann (2024, s. 4.2.3.) for more details.

One should note that rows containing \code{NA} values are removed from the panel before the testing procedure is performed.

Please be aware that the equivalence test based on the root mean squared placebo coefficient uses a randomization technique (as described by Dette & Schumann (2024)), leading to a stochastic critical value and minimum equivalence threshold. Therefore, the results may vary slightly between different runs of the function. For reproducibility, it is recommended to set a seed before using the function.
}
\examples{
# Generate a balanced panel dataset with 500 cross-sectional units (individuals), 
# 5 time periods (labeled 1-5), a binary variable indicating which individual 
# receives treatment and 2 control variables ("X_1" and "X_2"). 
# The error-terms are generated without  heteroscedasticity,  autocorrelation, 
# or any significant clusters. Furthermore, there are no fixed effects and 
# no pre-trends present in the data (all values in  beta are 0). 
# See sim_paneldata() for more details.

sim_data <- sim_paneldata(N = 500, tt = 5, p = 2, beta = rep(0, 5), 
                          gamma = rep(1, 2), het = 0, phi = 0, sd = 1, 
                          burnins = 50)

# Perform the equivalence test using an equivalence threshold of 1 with periods 
# 1-4 as pre-treatment periods based on the RMS testing procedure:
#  - option 1: using column names in the panel
# One can use the names of the columns in the panel to specify the variables:
rmsEquivTest(Y = "Y", ID = "ID", G = "G", period = "period", X = c("X_1", "X_2"),
             data = sim_data, equiv_threshold = 1, pretreatment_period = 1:4,
             base_period = 4)

#  - option 2: using column numbers in the panel 
# Alternatively, one can use the column numbers in the panel to specify the variables:
rmsEquivTest(Y = 3, ID = 1, G = 4, period = 2, X = c(5, 6),
             data = sim_data, equiv_threshold = 1, pretreatment_period = 1:4,
             base_period = 4)
             
#  - option 3: using separate variables 
# One can also use the variables directly without specifying the data variable:
data_Y <- sim_data$Y
data_ID <- sim_data$ID
data_G <- sim_data$G
data_period <- sim_data$period
data_X <- cbind(sim_data$X_1, sim_data$X_2)

rmsEquivTest(Y = data_Y, ID = data_ID, G = data_G, period = data_period, X = data_X,
             equiv_threshold = 1, pretreatment_period = 1:4,
             base_period = 4)

# The testing procedures can also be performed without specifying the 
# equivalence threshold specified. Then, the minimum equivalence threshold is returned
# for which the null hypothesis of non-negligible trend-differences can be rejected.
# Again, the three possible ways of entering the data as above can be used:
rmsEquivTest(Y = "Y", ID = "ID", G = "G", period = "period", X = c("X_1", "X_2"),
             data = sim_data, equiv_threshold = NULL, pretreatment_period = 1:4,
             base_period = 4)

rmsEquivTest(Y = 3, ID = 1, G = 4, period = 2, X = c(5, 6),
             data = sim_data, equiv_threshold = NULL, pretreatment_period = 1:4,
             base_period = 4)
             
rmsEquivTest(Y = data_Y, ID = data_ID, G = data_G, period = data_period, X= data_X,
             equiv_threshold = NULL, pretreatment_period = 1:4,
             base_period = 4)

# Finally, one should note that the test procedure also works for unbalanced panels.
# To illustrate this, we generate an unbalanced panel dataset by randomly selecting
# 70\% of the observations from the balanced panel dataset:

random_indeces <- sample(nrow(sim_data), 0.7*nrow(sim_data))
unbalanced_sim_data <- sim_data[random_indeces, ]
#  With Equivalence Threshold:
rmsEquivTest(Y = 3, ID = 1, G = 4, period = 2, X = c(5, 6),
             data = unbalanced_sim_data, equiv_threshold = 1, 
             pretreatment_period = 1:4, base_period = 4)

#  Without Equivalence Threshold:
rmsEquivTest(Y = 3, ID = 1, G = 4, period = 2, X = c(5, 6),
             data = unbalanced_sim_data, equiv_threshold = NULL, 
             pretreatment_period = 1:4, base_period = 4)


}
\references{
Dette, H., & Schumann, M. (2024). "Testing for Equivalence of Pre-Trends in Difference-in-Differences Estimation." \emph{Journal of Business & Economic Statistics}, 1–13. DOI: \doi{10.1080/07350015.2024.2308121}
}
\seealso{
\code{\link[=print.rmsEquivTest]{print.rmsEquivTest}}
}
\author{
Ties Bos
}
