% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cs-sequential.R
\name{cssrc}
\alias{cssrc}
\title{Cross-sectional sequential reconciliation-combination}
\usage{
cssrc(base, fc = "sa", comb = "ols", res = NULL, mse = TRUE, shrink = TRUE,
      nnw = FALSE, factorized = FALSE, ...)
}
\arguments{
\item{base}{A list of \eqn{p} numeric (\eqn{h \times n}) matrix or multivariate time series
(\code{mts} class) containing the base forecasts to be reconciled; \eqn{h} is the forecast
horizon, \eqn{n} is the total number of time series (\eqn{n = n_u + n_b}) and
\eqn{p} is the total number of experts.}

\item{fc}{A string specifying the combination method:
\itemize{
\item "\code{sa}" - (\emph{default}) simple average (equal weights).
\item "\code{var}" - (uses \code{res}) weights derived from the inverse
of forecasts variances/MSE as proposed by Bates and Granger (1969).
\item "\code{cov}" - (uses \code{res}) weights derived using the whole
forecast error covariance matrix, as proposed by Newbold and Granger (1974).
}}

\item{comb}{A string specifying the reconciliation method: \code{"ols"}, \code{"wls"},
\code{"shr"}, \code{"sam"} (see \href{https://danigiro.github.io/FoReco/}{\code{FoReco}}).}

\item{res}{A list of \eqn{p} numeric (\eqn{N \times n}) matrix containing the
in-sample residuals. This input is used to compute some covariance matrices.}

\item{mse}{If \code{TRUE} (\emph{default}) the residuals used to compute the covariance
matrix are not mean-corrected.}

\item{shrink}{If \code{TRUE} (\emph{default}), the covariance matrix
for \code{fc = "cov"} is shrunk.}

\item{nnw}{If \code{TRUE} for \code{fc = "cov"}, the weights are constrained to be
non-negative (Conflitti et al., 2015). The \emph{default} is \code{FALSE}.}

\item{factorized}{Value to be passed to the \code{\link[quadprog:solve.QP]{quadprog::solve.QP}},
only when \code{nnw = TRUE}.}

\item{...}{Arguments passed on to
\href{https://danigiro.github.io/FoReco/reference/csrec.html}{\code{FoReco::csrec}}
(e.g., \code{agg_mat} or \code{cons_mat}).}
}
\value{
A (\eqn{h \times n}) numeric matrix of cross-sectional combined and reconciled forecasts.
}
\description{
This function applies a sequential method that first reconciles the base forecasts
from each expert to satisfy the linear constraints, and then combines the reconciled
forecasts obtained so far. \link{cssrc} may be applied only in 'balanced' cases (e.g.,
\eqn{n_j = n} \eqn{\forall j}, see Girolimetto and Di Fonzo, 2024)
}
\examples{
set.seed(123)
# (2 x 3) base forecasts matrix (simulated), expert 1
base1 <- matrix(rnorm(6, mean = c(20, 10, 10)), 2, byrow = TRUE)
# (10 x 3) in-sample residuals matrix (simulated), expert 1
res1 <- t(matrix(rnorm(n = 30), nrow = 3))

# (2 x 3) base forecasts matrix (simulated), expert 2
base2 <- matrix(rnorm(6, mean = c(20, 10, 10)), 2, byrow = TRUE)
# (10 x 3) in-sample residuals matrix (simulated), expert 2
res2 <- t(matrix(rnorm(n = 30), nrow = 3))

# Base forecasts' and residuals' lists
base <- list(base1, base2)
res <- list(res1, res2)

# Aggregation matrix for Z = X + Y
A <- t(c(1,1))
reco <- cssrc(base = base, agg_mat = A, comb = "wls", res = res, fc = "sa")

# Zero constraints matrix for Z - X - Y = 0
C <- t(c(1, -1, -1))
reco <- cssrc(base = base, cons_mat = C, comb = "wls", res = res, fc = "sa") # same results

# WARNING!
reco_v <- cssrc(base = base, agg_mat = A, comb = "wls", res = res, fc = "var")
round(C \%*\% t(reco_v), 3) # Incoherent forecasts

}
\references{
Bates, J. and Granger, C. W. (1969), The combination of forecasts,
\emph{Operations Research Quarterly}, 20, 451–468. \doi{10.1057/jors.1969.103}.

Conflitti, C., De Mol, C., and Giannone, D. (2015), Optimal combination of survey
forecasts. \emph{International Journal of Forecasting}, 31(4), 1096–1103.
\doi{10.1016/j.ijforecast.2015.03.009}.

Girolimetto, D. and Di Fonzo, T. (2024), Coherent forecast combination for linearly
constrained multiple time series. \doi{10.48550/arXiv.2412.03429}.

Newbold, P. and Granger, C. W. (1974), Experience with forecasting
univariate time series and the combination of forecasts,
\emph{Journal of the Royal Statistical Society, A}, 137, 131–146.
\doi{10.2307/2344546}
}
\seealso{
Sequential coherent combination: 
\code{\link{csscr}()}
}
\concept{Sequential coherent combination}
