% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ForceAtlas2.R
\name{layout.forceatlas2}
\alias{layout.forceatlas2}
\title{Apply ForceAtlas2 layout to a graph}
\usage{
layout.forceatlas2(
  g,
  iterations = 100,
  linlog = FALSE,
  pos = NULL,
  gravity = 1,
  center = NULL,
  plotstep = 10,
  plotlabels = TRUE,
  scalingratio = 10,
  stronggravity = FALSE,
  jittertol = 1
)
}
\arguments{
\item{g}{An \code{igraph} object representing the graph.}

\item{iterations}{Integer. The number of iterations to run the algorithm. Default is 100.}

\item{linlog}{Logical. If \code{linlog = TRUE}, uses the Noack LinLog model implemented for `Gephi' to calculate attractive and repulsive forces (see Noack 2009). Default is \code{linlog = FALSE}.}

\item{pos}{A 2-column matrix of initial positions, where the columns contain x-coordinates and y-coordinates, respectively. If \code{pos = NULL}, positions for the first iteration are generated randomly. Default is \code{pos = NULL}.}

\item{gravity}{Numeric. The strength of the gravity force. Default is 1. Note that this is only included in calculations if \code{stronggravity = TRUE}. Higher gravity values result in tighter networks.}

\item{center}{A numeric vector of length 2 specifying the center of gravity. If \code{ center = NULL}, the center is calculated automatically. Default is \code{ center = NULL}.}

\item{plotstep}{Integer. The number of iterations between plots. If \code{plotstep = 0}, no plotting is done. Default is \code{plotstep = 10}. These plots appear as intermediate output in the console.}

\item{plotlabels}{Logical. If \code{plotlabels = TRUE}, plot node labels appear during the intermediate plotstep graphs. Default is \code{plotlabels = TRUE}.}

\item{scalingratio}{Numeric. The scaling ratio of the layout. Default is 10, in line with `Gephi'.}

\item{stronggravity}{Logical. If \code{stronggravity = TRUE}, gravity will be an additional force acting on each node, and the gravity parameter kicks in. Default is \code{stronggravity = FALSE}.}

\item{jittertol}{Numeric. The tolerance for jittering nodes. Default is \code{jittertol = 1};  Jacomy et al. (2014) do not recommend increasing it above 1.}
}
\value{
A matrix of node positions.
}
\description{
This function applies Jacomy et al. (2014)'s 'ForceAtlas2' layout algorithm to an \code{igraph} object.
}
\details{
This function implements Jacomy et al. (2014)'s ForceAtlas2 layout algorithm on an \code{igraph} object.
It can handle large graphs and is particularly suitable for visualizing networks. It also includes LinLog mode and a stronger gravity feature, like `Gephi'.
}
\examples{
# Create a random graph
library(igraph)
g <- sample_gnp(100, 0.05)

# Assign non-numeric row names
V(g)$name <- paste0("node", 1:vcount(g))

# Apply ForceAtlas2 layout
pos <- layout.forceatlas2(g, linlog = TRUE, iterations = 100, scalingratio = 10)

V(g)$x = as.numeric(pos[,1])
V(g)$y = as.numeric(pos[,2])

# Plotting (preserves the exact degree of rotation)
igraph::plot.igraph(g, rescale = FALSE,
                    xlim = range(pos[,1]), ylim = range(pos[,2]))

}
\references{
{
\insertRef{jacomy_forceatlas2_2014}{GephiForR} \cr
\insertRef{noack_modularity_2009}{GephiForR}
}
}
