% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/singleModalityAdmm.R
\name{singleModalityAdmm}
\alias{singleModalityAdmm}
\title{High-dimensional Single Modality Mediation Models}
\usage{
singleModalityAdmm(
  X,
  Y,
  M1,
  rho = 1,
  lambda1a,
  lambda1b,
  lambda1g,
  lambda2a,
  lambda2b,
  penalty = "ElasticNet",
  penaltyParameterList = list(),
  SIS = FALSE,
  SISThreshold = 2,
  maxIter = 3000L,
  tol = 0.001,
  verbose = FALSE,
  verboseOptions = list(numIter = 10L, numAlpha = 1L, numBeta = 1L, numGamma = 1L)
)
}
\arguments{
\item{X}{The matrix of independent variables (exposure/treatment/group).}

\item{Y}{The vector of dependent variable (outcome response).}

\item{M1}{The single-modality mediator.}

\item{rho}{The augmented Lagrangian parameter for ADMM.}

\item{lambda1a}{The L1-norm penalty for the effect between mediator and independent variables.}

\item{lambda1b}{The L1-norm penalty for the effect between mediator and dependent variable.}

\item{lambda1g}{The L1-norm penalty for the direct effect. Default is \strong{10} to adress overestimate issue.}

\item{lambda2a}{The L2-norm penalty for the effect between mediator and independent variables.
It's not used when Penalty is \code{PathwayLasso}.}

\item{lambda2b}{The L2-norm penalty for the effect between mediator and dependent variable.
It's not used when Penalty is \code{PathwayLasso}.}

\item{penalty}{A string to specify the penalty. Default is \code{ElasticNet}. Possible methods are
Elastic Net (\code{ElasticNet}), Pathway Lasso (\code{PathywayLasso}), and  Network-constrained Penalty (\code{Network}).}

\item{penaltyParameterList}{\itemize{
\item Penalty=\code{PathwayLasso} needs two parameters.
\itemize{
\item kappa The L1-norm penalty for pathway Lasso.
\item nu The L2-norm penalty for pathway Lasso.
}
\item Penalty=\code{Network} needs one parameter.
\itemize{
\item laplacianMatrix The Laplacian matrix applied on network penalty.
}
\item Penalty=\code{ElasticNet} don't need other parameters.
}}

\item{SIS}{A logical value to specify whether to perform sure independence screening (SIS).}

\item{SISThreshold}{The threshold value for the target reduced dimension for mediators. The default is "2," which reduces the dimension to 2*n/log(n).}

\item{maxIter}{The maximum iterations. Default is \code{3000}.}

\item{tol}{The tolerence of convergence threshold. Default is \code{1e-3}.}

\item{verbose}{A logical value to specify whether to print the iteration process.}

\item{verboseOptions}{A list of values:
\itemize{
\item \code{numIter}: The number of iterations to print.
\item \code{numAlpha}: The number of \code{alpha} to print.
\item \code{numBeta}: The number of \code{beta} to print.
\item \code{numGamma}: The number of \code{gamma} to print.
}}
}
\value{
A object, SingleModalityAdmm, with three elements.
\itemize{
\item gamma: estimated direct effect.
\item alpha: estimate effect between mediator and independent variables.
\item beta : estimate effect between mediator and dependent variable.
}
}
\description{
High-dimensional Single Modality Mediation Models
}
\examples{
## Generate Empirical Data
simuData <- modalityMediationDataGen(seed = 20231201, generateLaplacianMatrix = TRUE)

## Parameter Estimation for ElasticNet penalty
modelElasticNet <- singleModalityAdmm(
  X = simuData$MediData$X, Y = simuData$MediData$Y, M1 = simuData$MediData$M1,
  rho = 1, lambda1a = 1, lambda1b = 0.1, lambda1g = 2, lambda2a = 1, lambda2b = 1,
  penalty = "ElasticNet"
)

# fitted & predict
fitted(modelElasticNet)
predict(modelElasticNet, matrix(c(0, 1), ncol=1))

## Parameter Estimation for Pathway Lasso penalty
modelPathwayLasso <- singleModalityAdmm(
  X = simuData$MediData$X, Y = simuData$MediData$Y, M1 = simuData$MediData$M1,
  rho = 1, lambda1a = 1, lambda1b = 0.1, lambda1g = 2, lambda2a = 1, lambda2b = 1,
  penalty = "PathwayLasso", penaltyParameterList = list(kappa = 1, nu = 2)
)

## Parameter Estimation for Network penalty
modelNetwork <- singleModalityAdmm(
  X = simuData$MediData$X, Y = simuData$MediData$Y, M1 = simuData$MediData$M1,
  rho = 1, lambda1a = 1, lambda1b = 0.1, lambda1g = 2, lambda2a = 1, lambda2b = 1,
  penalty = "Network", penaltyParameterList = list(laplacianMatrix = simuData$Info$laplacianMatrix)
)

## Parameter Estimation for Network penalty with a customized Laplacian matrix
set.seed(20231201)
p <- ncol(simuData$MediData$M1)
W <- matrix(0, nrow = p, ncol = p)
W[lower.tri(W)] <- runif(p*(p-1)/2, 0, 1)
W[upper.tri(W)] <- t(W)[upper.tri(W)]
diag(W) <- 1
L <- weightToLaplacian(W)
modelNetwork <- singleModalityAdmm(
  X = simuData$MediData$X, Y = simuData$MediData$Y, M1 = simuData$MediData$M1,
  rho = 1, lambda1a = 1, lambda1b = 0.1, lambda1g = 2, lambda2a = 1, lambda2b = 1,
  penalty = "Network", penaltyParameterList = list(laplacianMatrix = L)
)

## With sure independence screening
## Generate Empirical Data
simuData <- modalityMediationDataGen(n = 50, p = 1000, seed = 20231201)

## Parameter Estimation for ElasticNet penalty
modelElasticNetSIS <- singleModalityAdmm(
  X = simuData$MediData$X, Y = simuData$MediData$Y, M1 = simuData$MediData$M1,
  rho = 1, lambda1a = 1, lambda1b = 0.1, lambda1g = 2, lambda2a = 1, lambda2b = 1,
  penalty = "ElasticNet", SIS = TRUE
)
fitted(modelElasticNetSIS)
predict(modelElasticNetSIS, matrix(c(0, 1), ncol=1))
}
