IC <- function(data, model, type){
  # Get sample size
  n <- length(data)

  # Determine model type and calculate parameters based on structure
  if(is.null(model$dwellparameters)){
    # Standard HMM model
    J <- length(model$estimate$delta)  # Number of states
    # Parameter count: transition probabilities + observation parameters
    params <- (J * (J - 1)) + (J * length(model$estimate[1:(length(model$estimate) - 2)]))
    loglik <- model$loglik
  } else {
    # HSMM model with dwell time parameters
    J <- length(model$delta)  # Number of states

    # Calculate parameter count based on dwell parameter structure
    if(all(model$dwellparameters$shift==1)){
      # Reduced parameters when all shifts equal 1
      params <- (J*(J-2)) + ((length(model$dwellparameters) - 1)*J) + (length(model$observationparameters)*J)
    } else {
      # Full parameter count
      params <- (J*(J-2)) + (length(model$dwellparameters)*J) + (length(model$observationparameters)*J)
    }
    # Use final log-likelihood value
    loglik <- model$loglikelihoods[length(model$loglikelihoods)]
  }

  # Calculate specified information criterion
  if(type=="AIC3"){
    # Modified AIC with penalty factor of 3
    IC <- -2*loglik + 3*params
  } else if(type=="AICu"){
    # Unbiased AIC (AICu) - corrected AIC with additional finite sample adjustment
    AIC <- -2*loglik + 2*params
    AICc <- AIC + ((2*params^2 + 2*params)/(n - params - 1))  # Finite sample correction
    IC <- AICc + n*log(n/(n - params - 1))  # Additional unbiased correction
  } else if(type=="HQC"){
    # Hannan-Quinn Criterion
    IC <- -2*loglik + 2*params*log(log(n))
  } else if(type=="BICadj"){
    # Adjusted Bayesian Information Criterion
    IC <- -2*loglik + params*log(n) - params*log(2*pi)
  } else {
    # Error for unsupported criterion types
    stop("Information criterion currently supported is : AIC3, AICu, HQC and BICadj. Please find AIC and BIC provided in model output.")
  }

  return(IC)
}
