\name{findmleHMM}
\alias{findmleHMM}
\title{Maximum Likelihood Estimation for Hidden Markov Models}
\description{
Estimates the parameters of a Hidden Markov Model (HMM) using either the EM-based semi-Markov parameterisation or direct numerical maximization of the likelihood.
}
\usage{
findmleHMM(J, x, obsdist, obspar, Pi, EM = FALSE, verbose = TRUE, seed = NULL, ...)
}
\arguments{
\item{J}{Integer. The number of hidden states in the HMM. Must be greater than 1.}
\item{x}{Numeric vector. The observed data sequence.}
\item{obsdist}{Character string. The observation distribution. Supported distributions are: "norm", "pois", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}
\item{obspar}{List. Initial parameters for the observation distribution. Required parameters vary by distribution:
  \itemize{
    \item \code{norm}: \code{mean}, \code{sd}
    \item \code{pois}: \code{lambda}
    \item \code{weibull}: \code{shape}, \code{scale}
    \item \code{zip}: \code{pi}, \code{lambda}
    \item \code{nbinom}: \code{size}, \code{mu}
    \item \code{zinb}: \code{pi}, \code{size}, \code{mu}
    \item \code{exp}: \code{rate}
    \item \code{gamma}: \code{shape}, \code{rate}
    \item \code{lnorm}: \code{meanlog}, \code{sdlog}
    \item \code{gev}: \code{loc}, \code{scale}, \code{shape}
    \item \code{ZInormal}: \code{mean}, \code{sd}, \code{pi}
    \item \code{ZIgamma}: \code{shape}, \code{rate}, \code{pi}
  }
  Each parameter should be a vector of length \code{J} with values for each state.
}
\item{Pi}{Matrix. The \code{J x J} transition probability matrix between states. Rows must sum to 1.}
\item{EM}{Logical. If \code{TRUE}, uses an EM-based semi-Markov approximation to estimate the HMM parameters. If \code{FALSE}, maximizes the likelihood directly using \code{nlm}. Defaults to \code{FALSE}.}
\item{verbose}{Logical. If \code{TRUE}, progress messages are printed to the console when \code{EM = TRUE}. Default is \code{TRUE}.}
\item{seed}{Integer or \code{NULL}. Random seed for reproducibility. Default is \code{NULL}.}
\item{...}{Further arguments to be passed in the case of \code{EM=TRUE}, such as \code{maxiter} and \code{tol}.}
}
\value{
A list containing:
\item{estimate}{List of estimated HMM parameters, including state-dependent observation parameters and transition probabilities.}
\item{loglik}{The maximized log-likelihood value.}
\item{AIC}{The Akaike Information Criterion for the fitted model.}
\item{BIC}{Bayesian Information Criteria for the fitted model.}
\item{hessian}{Optional. The Hessian matrix at the maximum likelihood estimates (returned if \code{EM = FALSE}).}
}
\details{
This function fits a Hidden Markov Model to a sequence of observations using either:
\itemize{
  \item An EM-based approximation via a semi-Markov model when \code{EM = TRUE}
  \item Direct numerical maximization of the likelihood using \code{nlm} when \code{EM = FALSE}
}
The EM-based approach initializes dwell-time probabilities from the diagonal of \code{Pi} and estimates the HMM via the corresponding semi-Markov model. After fitting, transition probabilities are adjusted to incorporate self-transition probabilities.
Supported observation distributions include normal, Poisson, Weibull, zero-inflated Poisson (ZIP), negative binomial, zero-inflated negative binomial (ZINB), exponential, gamma, log-normal, generalized extreme value (GEV), zero-inflated normal, and zero-inflated gamma. Parameter requirements vary by distribution.
When \code{verbose = TRUE} and \code{EM = TRUE}, progress messages from the EM algorithm are displayed.
}
\examples{
J <- 3
Pi <- matrix(c(0.8, 0.15, 0.05,
               0.1, 0.7, 0.2,
               0.2, 0.3, 0.5), nrow = 3, byrow = TRUE)

obspar <- list(
  mean = c(-2, 0, 3),
  sd = c(0.5, 1, 1.5)
)

x <- generateHMM(n = 200, J = J, obsdist = "norm",
                       obspar = obspar, Pi = Pi)$x

fit <- findmleHMM(J = J, x = x, obsdist = "norm",
                   obspar = obspar, Pi = Pi, EM = FALSE)

fit$estimate
fit$loglik
fit$AIC

fit_em <- findmleHMM(J = J, x = x, obsdist = "norm",
                      obspar = obspar, Pi = Pi, EM = TRUE, verbose = FALSE)
}
\seealso{
\code{\link{generateHMM}} for simulating HMM data.
\code{\link{findmleHSMM}} for EM-based semi-Markov estimation.
}
\author{
Aimee Cody
}


