\name{globaldecodeHSMM}
\alias{globaldecodeHSMM}
\title{Global Decoding of Hidden Semi-Markov Models}
\description{
Computes the most likely hidden state sequence for a univariate time series
using a fitted Hidden Semi-Markov Model (HSMM) via the Viterbi algorithm.
}
\usage{
globaldecodeHSMM(x, M = NA, HSMM, obsdist, dwelldist, shift = FALSE)
}
\arguments{
\item{x}{Numeric vector. The observed data sequence.}

\item{M}{Integer. Maximum dwell time to consider for semi-Markov states. Defaults to \code{min(length(x), 1000)} if \code{NA}.}

\item{HSMM}{A fitted HSMM object (output from \code{findmleHSMM}) containing the estimated model parameters.}

\item{obsdist}{Character string. Observation distribution. Supported distributions: "norm", "pois", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}

\item{dwelldist}{Character string. Dwell-time distribution. Supported distributions: "pois", "nbinom", "betabinom".}

\item{shift}{Logical character. Whether or not the dwell distribution has been shifted by a shift parameter. Defaults to FALSE.}

}
\value{
A numeric vector of length \code{length(x)}, giving the most likely hidden state at each observation.
}
\details{
This function computes the global most probable state sequence (Viterbi path) for a
sequence of observations under a Hidden Semi-Markov Model using the parameters contained
in a fitted HSMM object. The function calculates observation log-probabilities, dwell-time
log-probabilities, and forward log-probabilities to determine the Viterbi path.
}
\examples{
J <- 3
Pi <- matrix(c(0.0, 0.6, 0.4,
               0.5, 0.0, 0.5,
               0.3, 0.7, 0.0), nrow = J, byrow = TRUE)

obspar <- list(
  mean = c(4, 2, 6),
  sd   = c(1, 1, 2)
)

dwellpar <- list(
  lambda = c(3, 5, 4)
)

# Simulate HSMM data
sim <- generateHSMM(n = 1000, J = J, obsdist = "norm",
                    dwelldist = "pois", obspar = obspar,
                    dwellpar = dwellpar, Pi = Pi)

# Fit HSMM using the true parameters
HSMM_true <- findmleHSMM(x = sim$x, J = J, M = 100,
                          obsdist = "norm", dwelldist = "pois",
                          obspar = obspar, dwellpar = dwellpar,
                          Pi = Pi)

# Decode states using globaldecodeHSMM
states <- globaldecodeHSMM(x = sim$x, M = 100, HSMM = HSMM_true,
                           obsdist = "norm", dwelldist = "pois")

head(states)
}
\seealso{
\code{\link{findmleHSMM}}, for estimating HSMM parameters from data.
\code{\link{generateHSMM}}, for simulating HSMM data.
\code{\link{localdecodeHSMM}}, for local decoding of HSMM states.
\code{\link{globaldecodeHMM}}, for global decoding hidden Markov models.
}
\author{Aimee Cody}


