% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/node_icc.R
\encoding{UTF-8}
\name{node_icc}
\alias{node_icc}
\title{Interconnection centrality for connector nodes in a tripartite network}
\usage{
node_icc(network.or.subnet_mat1, subnet_mat2 = NULL, weighted = FALSE)
}
\arguments{
\item{network.or.subnet_mat1}{An igraph object or matrix. An "igraph" object with node attribute 'level' or a matrix representing one subnetwork. See details.}

\item{subnet_mat2}{A matrix representing one subnetwork.}

\item{weighted}{Logical. If TRUE, link strengths of connector nodes are used. Default to FALSE.}
}
\value{
Return a data frame with interconnection degree, betweenness and closeness for connector nodes.
}
\description{
Calculating interconnection centrality (degree, betweenness and closeness) specified for connector nodes in a tripartite network.
}
\details{
In this package, a tripartite network contains three groups of nodes (a-nodes,b-nodes,c-nodes) and two subnetworks (P includes the links between a-nodes and b-nodes, Q includes the links between b-nodes and c-nodes). Connector nodes belong to b-nodes.
This function calculates interconnection degree, betweenness and closeness centrality for connector nodes.
\itemize{
\item For binary networks, \strong{Interconnection degree} of each connector species is defined as the product of its degree values from two subnetworks.
\item \strong{Interconnection betweenness} is defined by the number of shortest paths from a-nodes to c-nodes going through connector species, \eqn{\sum_{\substack{i \in a,\, j \in c \\ i \neq j}} \frac{g_{ivj}}{g_{ij}}},
where \eqn{g_{ij}} is the total number of shortest paths between node  \eqn{i} from a-nodes and \eqn{j} from c-nodes while \eqn{g_{ivj}} is the number of those shortest paths which pass through connector species \eqn{v}.
\item \strong{Interconnection closeness} is defined by the inverse of the sum of distances from connector species to both a-nodes and c-nodes, \eqn{1/(\sum_{v \neq i, i \in a \cup c} d_{vi})},
where \eqn{d_{vi}} is the distance between connector species \eqn{v} and species \eqn{i} from a-nodes and c-nodes.
For weighted networks, interaction strengths are used in the calculation of weighted degree, shorest path, and distance.
}
\strong{network.or.subnet_mat1 and subnet_mat2}

Two types of inputs \code{network.or.subnet_mat1} can be processed:
\itemize{
\item An "igraph" object with node attribute 'level' (0 for a-nodes, 1 for b-nodes,2 for c-nodes). If the input is a weighted network, the edge should have a 'weight' attribute.
\item Or a matrix representing subnetwork P, and must be input with \code{subnet_mat2} representing subnetwork Q.
}

If the inputs are two matrices, please make sure the rows of
\code{network.or.subnet_mat1} and \code{subnet_mat2} both represent the groups of connector species,i.e, the b-group species. If both matrices have row names, then the function matches row
names to produce connector nodes. Otherwise, row numbers are assigned as row names.
}
\examples{

## generate a random tripartite network
set.seed(12)
Net <- build_toy_net(11,15,16,0.2)

data(PPH_Coltparkmeadow)
Net <- PPH_Coltparkmeadow
node_icc(Net)
set.seed(13)
library(igraph)
E(Net)$weight<-runif(length(E(Net)),0.1,1)#random weights assigned
node_icc(Net,weighted=TRUE)

##input as binary matrices,with row names.
set.seed(12)
md1 <- matrix(sample(c(0,1),5*8,replace=TRUE),5,8,
       dimnames = list(paste0("b",1:5),paste0("c",1:8)))
md2 <- matrix(sample(c(0,1),20*30,replace=TRUE),20,30,
       dimnames = list(paste0("b",1:20),paste0("a",1:30)))
node_icc(md1,md2)

##input as weighted matrices,with row numbers as row names.
set.seed(17)
mdw1 <- matrix(sample(c(rep(0,20),runif(20,0,1))),5,8)
mdw2 <- matrix(sample(c(rep(0,500),runif(100,0,1))),20,30)
node_icc(mdw1,mdw2)
node_icc(mdw1,mdw2,weighted=TRUE)

}
