#' Provides analysis results related to the I-SPY ARDS composite ranking.
#' @param TRT Treatment vs control indicator Vector (1s and 0s).
#' @param Time_To_Death Vector containing time to death for participants from study entry.
#' @param Days_On_Ventilator Vector containing the number of days on ventilator for participants within the first 28 days of study enrollment. The I-SPY ARDS trial counts days on ventilator as the number of days from study entry until the day that ventilator is discontinued for good in the first 28 days.
#' @param Days_On_Advanced_O2 Vector containing the number of days that each participant was on advanced respiratory support within the first 28 days of study enrollment.
#' @return A list of 4 results based on the hierarchical composite ARDS ranking including: (1) The posterior probability and p-value used for graduation, (2) The Bayesian Wilcoxon/Mann-Whitney U test results along with posterior summaries on the Win Probability, (3) Frequentist summaries of the Win Probability, Win Ratio, and 2 sided Wilcoxon P-value, and (4) Component-wise interpretation of the composite win-ranking. This includes frequentist comparisons of the 28 day mortality rate, ventilator days in survivors, how many ties there were based on mortality and ventilation, and the advanced respiratory day distribution in tied patients.
#' @importFrom DFBA dfba_mann_whitney
#' @examples
#' ###load("Example_ARDS.rda")
#' library(DFBA)
#' Get_ARDS_Win_Probability(Example_Data$Treatment,Example_Data$Time_To_Death,
#' Example_Data$Days_On_Ventilator,Example_Data$Days_On_Advanced_O2)
#'@export
Get_ARDS_Win_Probability = function(TRT, ###Treatment vs control indicator (1s and 0s)
                                    Time_To_Death, ###Time to death for participants from study entry
                                    Days_On_Ventilator, ###Days on ventilator for participants since study entry. The I-SPY ARDS trial counts days on ventilator as the number of days from study entry until the day that ventilator is discontinued for good in the first 28 days.
                                    Days_On_Advanced_O2){


  ##Formatting some things to read into code easier
  DEATH = Time_To_Death<=28
  DEATH[is.na(DEATH)]=0
  COMPOSITE_RANKING = rep(NA,length(DEATH))
  DAYS_VENTILATION = Days_On_Ventilator
  DAYS_ADVANCED = Days_On_Advanced_O2
  N=length(TRT)


  ###Start building the composite ranking based on the described hierarchy
  COMPOSITE_RANKING[which(DEATH==1)]=-1 ##Set to -1 just to avoid any issues with the ranks after this
  #2. The best outcome is 0 days on invasive mechanical ventilation or ECMO and no more than 1 day
  #on advanced respiratory support.
  COMPOSITE_RANKING[which(DAYS_VENTILATION==0 & DAYS_ADVANCED<=1 & is.na(COMPOSITE_RANKING))]=N


  #3. First, subjects are ranked according to the number of days on invasive mechanical ventilation or
  #ECMO.
  #4. Within the categories set in #3, subjects are then ranked according to number of days on
  #advanced respiratory support (nested ranking).

  which_no_best_worst_outcome_composite = which(is.na(COMPOSITE_RANKING))
  VENTILATOR_RANK = rank(28-DAYS_VENTILATION[which_no_best_worst_outcome_composite],ties.method = "average")

  COMPOSITE_RANKING_NOT_BEST_WORST = VENTILATOR_RANK ###Storing this
  ##Break ventilator rank ties
  UN_VENTILATOR_RANK = unique(VENTILATOR_RANK)
  for(i in 1:length(UN_VENTILATOR_RANK)){
    which1 = which(VENTILATOR_RANK==UN_VENTILATOR_RANK[i])
    if(length(which1)>1){
      ###these are all tied, can we re-rank based on Advanced support?
      TIE_BREAK_DAYS_ADVANCED =   sort(unique(DAYS_ADVANCED[which_no_best_worst_outcome_composite[which1]]))

      GRID =  .001*(1:length(TIE_BREAK_DAYS_ADVANCED))/length(TIE_BREAK_DAYS_ADVANCED)

      ###Creates a grid for tie-breaks
      ##We will subtract each value of the grid
      ##in order of days on advanced respiratory support
      ##Goal is to get the composite ranking order

      for(j in 1:length(TIE_BREAK_DAYS_ADVANCED)){
        ###Look through the order of unique tie break days in order, subtract the corresponding grid value from the ventilator score
        which2 = which(DAYS_ADVANCED[which_no_best_worst_outcome_composite[which1]] == TIE_BREAK_DAYS_ADVANCED[j])

        ###All these patients had this # of days on advanced support
        VENTILATOR_RANK[which1[which2]]=VENTILATOR_RANK[which1[which2]]-GRID[j]
      }

    }
  }




  COMPOSITE_RANKING[which_no_best_worst_outcome_composite]=VENTILATOR_RANK


  ###Now we can re-rank again.
  COMPOSITE_RANKING = rank(COMPOSITE_RANKING,ties.method = "average")





  win_count = 0
  win_control = 0
  win_tie = 0
  for(i1 in which(TRT==1)){
    for(i2 in which(TRT==0)){
      if(COMPOSITE_RANKING[i1]>COMPOSITE_RANKING[i2]){
        win_count = win_count+1
      }
      if(COMPOSITE_RANKING[i2]>COMPOSITE_RANKING[i1]){
        win_control = win_control+1
      }
      if(COMPOSITE_RANKING[i2]==COMPOSITE_RANKING[i1]){
        win_tie = win_tie+1
      }
    }
  }



  FREQUENTIST_WR = rep(NA,3)
  FREQUENTIST_WR[1]=(win_count+.5*win_tie)/(win_count+win_tie+win_control)
  FREQUENTIST_WR[2]=win_count/win_control
  FREQUENTIST_WR[3]=wilcox.test(COMPOSITE_RANKING~TRT)$p.value
  names(FREQUENTIST_WR) = c("Win Probability","Win Ratio","2-Sided Wilcoxon P-value")

  m1 =   dfba_mann_whitney(COMPOSITE_RANKING[TRT==1], COMPOSITE_RANKING[TRT==0],hide_progress = FALSE)


  RESULTS= as.list(rep(NA,4))
  names(RESULTS) = c("Graduation Criteria","Bayesian Results","Frequentist Results","Component-Wise Interpretation")


  ##Store Graduation Criteria
  X1 = table(TRT,DEATH)
  FISHER_ONE_SIDED = fisher.test(X1,alternative="less")$p.value
  BAYES_WILCOX = m1$prH1
  GRADUATION = c(BAYES_WILCOX,FISHER_ONE_SIDED)
  names(GRADUATION) = c("Bayesian Wilcoxon on Composite","One-sided Fisher on Mortality")

  RESULTS[[1]]=GRADUATION
  RESULTS[[2]]=m1
  RESULTS[[3]]=FREQUENTIST_WR

  ###Fill in the interpretations
  HOLD = matrix(nrow=6,ncol=3)
  colnames(HOLD)=c("Treatment","Control","P-value")
  rownames(HOLD)=c("28 Day Mortality %","Average (SD) Ventilator Days in Survivors",
                   "0 Ventilator Days in Survivors %",
                   "28 or More Day Ventilator Days in Survivors %",
                   "# Ties on Mortality and Ventilator Days",
                   "Average (SD) Advanced Respiratory Days in Ties")



  ###Fill in Mortality First
  HOLD[1,1]=round(mean(DEATH[which(TRT==1)]),3)*100
  HOLD[1,2]=round(mean(DEATH[which(TRT==0)]),3)*100
  pval1 = round(fisher.test(table(DEATH,TRT))$p.value,3)
  if(pval1==0){pval1="<.001"}
  HOLD[1,3]=pval1


  ###Ventilator days in survivors
  HOLD[2,1]=paste0(round(mean(DAYS_VENTILATION[which(TRT==1 & DEATH==0)]),1)," (",round(sd(DAYS_VENTILATION[which(TRT==1 & DEATH==0)]),1),")")
  HOLD[2,2]=paste0(round(mean(DAYS_VENTILATION[which(TRT==0 & DEATH==0)]),1)," (",round(sd(DAYS_VENTILATION[which(TRT==0 & DEATH==0)]),1),")")
  pval1=round(wilcox.test(DAYS_VENTILATION[DEATH==0]~TRT[DEATH==0])$p.value,3)
  if(pval1==0){pval1="<.001"}
  HOLD[2,3]=pval1



  ###Get rate of 0 day ventilation
  HOLD[3,1]=round(mean(DAYS_VENTILATION[which(TRT==1 & DEATH==0)]<=0),3)*100
  HOLD[3,2]=round(mean(DAYS_VENTILATION[which(TRT==0 & DEATH==0)]<=0),3)*100
  pval1 = round(fisher.test(table(DAYS_VENTILATION[DEATH==0]<=0,TRT[DEATH==0]))$p.value,3)
  if(pval1==0){pval1="<.001"}
  HOLD[3,3]=pval1

  ###Get rate of 28 day ventilation
  HOLD[4,1]=round(mean(DAYS_VENTILATION[which(TRT==1 & DEATH==0)]>=28),3)*100
  HOLD[4,2]=round(mean(DAYS_VENTILATION[which(TRT==0 & DEATH==0)]>=28),3)*100
  pval1 = round(fisher.test(table(DAYS_VENTILATION[DEATH==0]>=28,TRT[DEATH==0]))$p.value,3)
  if(pval1==0){pval1="<.001"}
  HOLD[4,3]=pval1

  ###Find all ties
  IS_TIE = rep(0,length(TRT))
  for(i in which(TRT==1)){
    if(DEATH[i]==0 & DAYS_VENTILATION[i]<28){
      which_control = which(TRT==0 & DAYS_VENTILATION==DAYS_VENTILATION[i])
      if(length(which_control)>0){
        IS_TIE[which_control]=1
        IS_TIE[i]=1
      }
    }
  }


  ###How many ties
  HOLD[5,1]=sum(IS_TIE[TRT==1],na.rm=TRUE)
  HOLD[5,2]=sum(IS_TIE[TRT==0],na.rm=TRUE)
  HOLD[5,3]=""

  ##LAstly do advanced days in the ties
  HOLD[6,1]=paste0(round(mean(DAYS_ADVANCED[TRT==1 & IS_TIE==1]),1)," (", round(sd(DAYS_ADVANCED[TRT==1 & IS_TIE==1]),1),")")
  HOLD[6,2]=paste0(round(mean(DAYS_ADVANCED[TRT==0 & IS_TIE==1]),1)," (", round(sd(DAYS_ADVANCED[TRT==0 & IS_TIE==1]),1),")")
  pval1 = round(wilcox.test(DAYS_ADVANCED[IS_TIE==1]~TRT[IS_TIE==1])$p.value,3)
  if(pval1==0){pval1="<.001"}
  HOLD[6,3]=pval1

  RESULTS[[4]]=HOLD

  return(RESULTS)
}
