# JapanAPIs - Access Japanese Data via Public APIs and Curated Datasets
# Version 0.1.1
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_japan_life_expectancy

library(testthat)

test_that("get_japan_life_expectancy() returns a tibble with expected structure", {
  skip_on_cran()

  result <- get_japan_life_expectancy()

  # Must return NULL or tibble
  expect_true(is.null(result) || inherits(result, "tbl_df"))

  if (is.null(result)) return(invisible(TRUE))

  # Structure
  expect_s3_class(result, "tbl_df")
  expect_equal(ncol(result), 4)

  # Column names
  expect_named(result, c("indicator", "country", "year", "value"))

  # Column types
  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_type(result$value, "double")
})

test_that("get_japan_life_expectancy() returns correct years and row count", {
  skip_on_cran()

  result <- get_japan_life_expectancy()
  if (is.null(result)) return(invisible(TRUE))

  # Should return 2010–2022 (13 rows)
  expect_equal(nrow(result), 13)

  # Years must be correct and sorted descending
  expect_equal(sort(result$year), 2010:2022)
  expect_true(all(diff(result$year) <= 0))
})

test_that("get_japan_life_expectancy() contains valid and consistent values", {
  skip_on_cran()

  result <- get_japan_life_expectancy()
  if (is.null(result)) return(invisible(TRUE))

  # Country must always be Japan
  expect_true(all(result$country == "Japan"))

  # Indicator text must always match
  expect_true(all(grepl("Life expectancy", result$indicator, ignore.case = TRUE)))

  # Values must be non-negative
  expect_true(all(result$value >= 0, na.rm = TRUE))

  # Japan life expectancy is historically between 80 and 90
  expect_true(all(result$value >= 80, na.rm = TRUE))
  expect_true(all(result$value <= 90, na.rm = TRUE))
})

test_that("get_japan_life_expectancy() returns no missing essential values", {
  skip_on_cran()

  result <- get_japan_life_expectancy()
  if (is.null(result)) return(invisible(TRUE))

  # No NA values expected in life expectancy
  expect_true(all(!is.na(result$value)))
  expect_true(all(!is.na(result$year)))
  expect_true(all(!is.na(result$country)))
  expect_true(all(!is.na(result$indicator)))
})

test_that("get_japan_life_expectancy() handles API errors gracefully", {
  skip_on_cran()

  result <- tryCatch(get_japan_life_expectancy(), error = function(e) NULL)

  # Must return NULL or tibble
  expect_true(is.null(result) || inherits(result, "tbl_df"))
})
