#' A-Efficiency for Nested Partially Balanced Bipartite Block (NPBBB) Designs
#'@description
#'This function calculates the A-efficiency values for the block and sub-block structures of a given NPBBB design
#'@export
#'@usage
#'NPBBBD_Aeff(bd, sbd, v1, v2)
#'@param bd Block design (ignoring sub-blocks) in matrix form considering rows as blocks
#'@param sbd Sub-block design (ignoring blocks) in matrix form considering rows as blocks
#'@param v1 Number of test treatments
#'@param v2 Number of control treatments
#'@returns The output includes:
#' \itemize{
#'   \item \code{v1}: Number of test treatments
#'   \item \code{v2}: Number of controls
#'   \item \code{b1}: Number of blocks
#'   \item \code{b2}: Number of sub-blocks
#'   \item \code{r1}: Test replications
#'   \item \code{r2}: Control replications
#'   \item \code{k1}: Block size
#'   \item \code{k2}: Sub-block size
#'   \item \code{Lambdas1}: Lambda values from block design
#'   \item \code{Lambdas2}: Lambda values from sub-block design
#'   \item \code{E1}: Block design efficiency
#'   \item \code{E2}: Sub-block design efficiency
#' }
#'@references Vinayaka, Parsad R, Mandal BN, Dash S, LN Vinaykumar, Kumar M, Singh DR (2023) <doi:10.1080/03610926.2023.2251623>
#'@examples
#' ## Not run:
#' library(NPBBBDAefficiency)
#' NPBBBD_Aeff(matrix(c(1, 2, 3, 4), nrow = 2, byrow = TRUE),
#'             matrix(c(1, 2, 3, 4), nrow = 2, byrow = TRUE),
#'             v1 = 2, v2 = 2)
#' ## End(Not run)

NPBBBD_Aeff <- function(bd, sbd, v1, v2) {
  # bd is block design
  # sbd is sub-block design
  # v1 is test treatments
  # v2 is controls
  # Helper function to calculate MinTrace
  MinTrace <- function(v1, v2, b, k, r2) {
    Q1 <- r2^2 / b
    temp <- k * r2 - v2 * Q1
    t1 <- v1 * k / temp
    t2 <- v1 * (v2 - 1) / r2
    temp <- b * v1 * k * (k - 1) - (v1 * (k - 1) + k) * v2 * r2 + Q1 * v2^2
    t3 <- (v1 * v2 * k * (v1 - 1)^2) / temp
    return(t1 + t2 + t3)
  }

  # Helper function to calculate Trace
  Trace <- function(v1, v2, b, k, N) {
    rvec <- N %*% matrix(1, b, 1)
    v <- v1 + v2
    R <- diag(as.vector(rvec))
    NNP <- N %*% t(N)
    C <- R - NNP / k
    A <- C[1:v1, 1:v1]
    D <- C[(v1 + 1):v, (v1 + 1):v]
    f1 <- sum(diag(A)) / v1
    f4 <- sum(diag(D)) / v2
    f2 <- (sum(A) - v1 * f1) / (v1 * (v1 - 1))
    f5 <- (sum(D) - v2 * f4) / (v2 * (v2 - 1))
    return(v2 * (v1 - 1) / (f1 - f2) + v1 * (v2 - 1) / (f4 - f5) + v2 / (f1 + (v1 - 1) * f2))
  }

  # Helper function to calculate design_to_N
  design_to_N <- function(design) {
    v = max(design)
    b = nrow(design)
    k = ncol(design)
    N = matrix(0, v, b)
    for (i in 1:b) {
      for (j in 1:k) {
        N[design[i, j], i] = N[design[i, j], i] + 1
      }
    }
    return(N)
  }

  # Main logic for block design efficiency
  b1 <- nrow(bd)
  k1 <- ncol(bd)
  N1 <- design_to_N(as.matrix(bd))
  NNP1 <- N1 %*% t(N1)
  rvec1 <- N1 %*% matrix(1, b1, 1)
  r2<- floor(mean(rvec1[(v1 + 1):(v1 + v2)]))
  trd1 <- Trace(v1, v2, b1, k1, N1)
  mintrace1 <- MinTrace(v1, v2, b1, k1, r2)
  E1 <- mintrace1 / trd1

  # Main logic for sub-block design efficiency
  b2 <- nrow(sbd)
  k2 <- ncol(sbd)
  N2 <- design_to_N(as.matrix(sbd))
  NNP2 <- N2 %*% t(N2)
  rvec2 <- N2 %*% matrix(1, b2, 1)
  r2 <- floor(mean(rvec2[(v1 + 1):(v1 + v2)]))
  trd2 <- Trace(v1, v2, b2, k2, N2)
  mintrace2 <- MinTrace(v1, v2, b2, k2, r2)
  E2 <- mintrace2 / trd2

  # Output Results
  results <- list(
    "test treatments" = v1,
    "controls" = v2,
    "blocks" = b1,
    "sub-blocks" = b2,
    "test replications"=min(rowSums(N1)),
    "controls replications"=r2,
    "block size" = k1,
    "sub-block size" = k2,
    "Lambdas1" = unique(NNP1[row(NNP1) != col(NNP1)]),
    "Lambdas2" = unique(NNP2[row(NNP2) != col(NNP2)]),
    "Block design efficiency" = E1,
    "Sub-block design efficiency" = E2
  )
  return(results)
}
