#' Compute Rank-based Stability Index (RSI)
#'
#' @param data A data frame with genotype, environment, and response columns.
#' @param y Response variable column.
#' @param gen Genotype column.
#' @param env Environment column.
#' @param rep Optional replication column.
#'
#' @return A table containing RSI values for each genotype.
#' @references
#' 1. Kumar, P., Paul, A. K., Paul, R. K., Raju, B., Rathod, S., Ray, M., ... & Yeasin, M. (2024).
#'    A robust non-parametric stability measure to select stable genotypes.
#'    *The Indian Journal of Agricultural Sciences*, 94(9), 1007–1012.
#' 2. Pour-Aboughadareh, A., Khalili, M., Poczai, P., & Olivoto, T. (2022).
#'    Stability Indices to Deciphering the Genotype-by-Environment Interaction (GEI) Effect:
#'    An Applicable Review for Use in Plant Breeding Programs. *Plants*, 11(3), 414.
#'
#' @examples
#' library(NPStability)
#' library(dplyr)
#' data(example_data)
#' rsi_results <- compute_rsi(example_data, Yield, Gen, Env)
#' rsi_biplot <- rsi_biplot(example_data, Yield, Gen, Env, rsi_table = rsi_results)
#' @import dplyr
#' @import tidyr
#' @import rlang
#' @import ggplot2
#'
#' @importFrom dplyr %>%
#' @export
compute_rsi <- function(data, y, gen, env, rep = NULL) {

  GE <- gmean <- emean <- vij <- abs_vij <- rij <- RS <- S2 <- rRS <- rS2 <- RSI_value <- rRSI <- NULL

  yq   <- rlang::enquo(y)
  genq <- rlang::enquo(gen)
  envq <- rlang::enquo(env)

  d <- as_tibble(data) %>%
    dplyr::mutate(
      !!genq := as.factor(!!genq),
      !!envq := as.factor(!!envq)
    )

  ge <- d %>%
    dplyr::group_by(!!genq, !!envq) %>%
    dplyr::summarise(GE = mean(!!yq, na.rm = TRUE), .groups = "drop")

  mu <- mean(ge$GE, na.rm = TRUE)

  gen_name <- rlang::as_name(genq)
  env_name <- rlang::as_name(envq)

  gmeans <- ge %>%
    dplyr::group_by(!!genq) %>%
    dplyr::summarise(gmean = mean(GE, na.rm = TRUE), .groups = "drop") %>%
    dplyr::mutate(alpha = gmean - mu)

  emeans <- ge %>%
    dplyr::group_by(!!envq) %>%
    dplyr::summarise(emean = mean(GE, na.rm = TRUE), .groups = "drop") %>%
    dplyr::mutate(beta = emean - mu)

  ge2 <- ge %>%
    dplyr::left_join(gmeans, by = gen_name) %>%
    dplyr::left_join(emeans, by = env_name) %>%
    dplyr::mutate(
      vij = GE - mu - alpha - beta,
      abs_vij = abs(vij)
    )

  ge2 <- ge2 %>%
    dplyr::group_by(!!envq) %>%
    dplyr::mutate(rij = dplyr::min_rank(abs_vij)) %>%
    dplyr::ungroup()

  rs <- ge2 %>%
    dplyr::group_by(!!genq) %>%
    dplyr::summarise(RS = sum(rij, na.rm = TRUE),
                     mean_GE = mean(GE, na.rm = TRUE),
                     .groups = "drop")

  s2 <- ge2 %>%
    dplyr::group_by(!!genq) %>%
    dplyr::summarise(S2 = stats::var(abs_vij, na.rm = TRUE), .groups = "drop")

  out <- rs %>%
    dplyr::left_join(s2, by = gen_name) %>%
    dplyr::mutate(
      rRS = dplyr::min_rank(RS),
      rS2 = dplyr::min_rank(S2),
      RSI_value = (rRS + rS2) / 2,
      rRSI = dplyr::min_rank(RSI_value)
    ) %>%
    dplyr::arrange(rRSI)

  return(as_tibble(out))
}
