# InvGenGPZ.R
#' Inverse Generalized Gompertz Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Inverse Generalized Gompertz distribution.
#'
#' The Inverse Generalized Gompertz distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Inverse Generalized Gompertz distribution has CDF:
#'
#'\deqn{
#' F(x; \alpha, \lambda, \theta) =
#' 1 - \left[ 1 - \exp\left( \frac{\lambda}{\alpha} \left( 1 - \exp(\alpha / x) \right) \right) \right]^{\theta},
#' \quad x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' The implementation includes the following functions:
#' \itemize{
#'   \item \code{dinv.gen.gpz()} — Density function
#'   \item \code{pinv.gen.gpz()} — Distribution function
#'   \item \code{qinv.gen.gpz()} — Quantile function
#'   \item \code{rinv.gen.gpz()} — Random generation
#'   \item \code{hinv.gen.gpz()} — Hazard function
#' }
#'
#' @usage
#' dinv.gen.gpz(x, alpha, lambda, theta, log = FALSE)
#' pinv.gen.gpz(q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qinv.gen.gpz(p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rinv.gen.gpz(n, alpha, lambda, theta)
#' hinv.gen.gpz(x, alpha, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dinv.gen.gpz}: numeric vector of (log-)densities
#'   \item \code{pinv.gen.gpz}: numeric vector of probabilities
#'   \item \code{qinv.gen.gpz}: numeric vector of quantiles
#'   \item \code{rinv.gen.gpz}: numeric vector of random variates
#'   \item \code{hinv.gen.gpz}: numeric vector of hazard values
#' }
#'
#' @references  
#' Chaudhary, A.K., & Kumar, V. (2017). 
#' Inverse Generalized Gompertz Distribution with Properties and Applications.
#' \emph{Journal of National Academy of Mathematics}, \bold{31}, 1--15.  
#'
#' @examples
#' x <- seq(2, 5, 0.25)
#' dinv.gen.gpz(x, 1.5, 2.5, 5.0)
#' pinv.gen.gpz(x, 1.5, 2.5, 5.0)
#' qinv.gen.gpz(0.5, 1.5, 2.5, 5.0)
#' rinv.gen.gpz(10, 1.5, 2.5, 5.0)
#' hinv.gen.gpz(x, 1.5, 2.5, 5.0)
#' 
#' # Data
#' x <- fibers63
#' # ML estimates
#' params = list(alpha=3.4106, lambda=5.4685, theta=20.9199)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pinv.gen.gpz, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qinv.gen.gpz, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dinv.gen.gpz, pfun=pinv.gen.gpz, plot=TRUE)
#' print.gofic(out)
#' 
#' @name InvGenGPZ
#' @aliases dinv.gen.gpz pinv.gen.gpz qinv.gen.gpz rinv.gen.gpz hinv.gen.gpz
#' @export
NULL

#' @rdname InvGenGPZ
#' @usage NULL
#' @export   
dinv.gen.gpz <- function (x, alpha, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, lambda, theta) <= 0)) {
        stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- exp(alpha/x[valid])	
	z <- 1- u 	
	tmp <- 1- exp((lambda/alpha) * z )	
    pdf[valid] <- lambda * theta * (u/(x[valid]*x[valid])) * exp((lambda/alpha) * z ) * (tmp^(theta-1))
    }
	if (log) 
        pdf <- log(pdf)
    return(pdf)   
}

#' @rdname InvGenGPZ
#' @usage NULL
#' @export 
pinv.gen.gpz <- function (q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda, theta) <= 0)) {
		stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- exp(alpha/q[valid])		
	z <- 1- exp((lambda/alpha) * (1-u))
    cdf[valid] <- 1.0 - z ^ theta
    }
	if (!lower.tail) 
        cdf <- 1.0 - cdf 
    if (log.p) 
        cdf <- log(cdf)    
    return(cdf)   
}

#' @rdname InvGenGPZ
#' @usage NULL
#' @export 
qinv.gen.gpz <- function (p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda' and 'theta' must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	tmp <- log(1.0 - (alpha/lambda)*log(1.0 - (1-p[valid])^(1/theta)))
    qtl[valid] <-  alpha * (1/tmp)
	}
    return(qtl)   
}

#' @rdname InvGenGPZ
#' @usage NULL
#' @export 
rinv.gen.gpz <- function(n, alpha, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda, theta), is.numeric)))
		stop("'alpha', 'lambda', and 'theta' must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qinv.gen.gpz(u, alpha, lambda, theta)
    return(q)
}

#' @rdname InvGenGPZ
#' @usage NULL
#' @export 
hinv.gen.gpz <- function (x, alpha, lambda, theta)
{
	if (!all(sapply(list(x, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
	nume <- dinv.gen.gpz(x, alpha, lambda, theta) 
	surv <- 1.0 - pinv.gen.gpz(x, alpha, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)   
} 

