#' @title EM-Delta
#' @description
#' Computes a confidence interval for the OVL between two populations under Gaussian and two-component Gaussian mixture models, or both
#' populations modeled as two-component Gaussian mixtures, using EM-based estimation and the delta method.
#' @param x Numeric vector. Data from the first group. When \code{all_mix = FALSE}, this group
#'   is modeled as Gaussian.
#' @param y Numeric vector. Data from the second group, modeled as a two-component Gaussian mixture.
#' @param alpha confidence level.
#' @param h Step size used to compute numerical derivatives.
#' @param interv Numeric vector of length 2. Search interval for intersection points between the
#'   corresponding densities.
#' @param all_mix Logical. If \code{TRUE}, both groups are modeled as two-component Gaussian mixtures.
#'   If \code{FALSE}, only \code{y} is modeled as a mixture and \code{x} is Gaussian.
#'
#' @return A list containing a confidence interval.
#' Additional elements (e.g., \code{var_OVL}, parameter estimates, \code{OVL_hat}) may also be returned.
#'
#' @examples
#' set.seed(1)
#' x <- ifelse(runif(100) < 0.5, rnorm(100, mean = 0, sd = 1), rnorm(100, mean = 2, sd = 1))
#' y <- ifelse(runif(100) < 0.5, rnorm(100, mean = 2.5, sd = 1), rnorm(100, mean = 2, sd = 1))
#'res <- OVL.Delta.mix(x, y, all_mix = TRUE, interv = c(-10, 10))
#'res$IC1
#'res$IC2

#'
#' @export OVL.Delta.mix
#' @importFrom stats dnorm pnorm qnorm sd
#' @importFrom Matrix bdiag
#' @importFrom mixtools normalmixEM

OVL.Delta.mix <- function(x, y, alpha = 0.05, h = 10^(-5), interv = c(0, 20), all_mix = FALSE){

  if(all_mix == FALSE){
    n1 <- length(x)
    n2 <- length(y)
    mu1_hat <- mean(x)
    sigma1_hat <- sd(x) * sqrt((length(x) - 1) / (length(x)))

    EM_result <- try(normalmixEM(y), silent = TRUE)
    if (inherits(EM_result, "try-error")) {
      EM_result <- EM(y)
      mu2_hat <- EM_result$mu
      sigma2_hat <- EM_result$sigma
      pi2_hat <- EM_result$pi
    } else {
      mu2_hat <- EM_result$mu
      sigma2_hat <- EM_result$sigma
      pi2_hat <- EM_result$lambda
    }

    initial_params <- list(pi = pi2_hat[1], mu1 = mu2_hat[1], mu2 = mu2_hat[2],
                           sigma1 = sigma2_hat[1], sigma2 = sigma2_hat[2])

    FIM_Y <- try(FIM_mixture_normals(data = y, params = initial_params), silent = TRUE)
    if (inherits(FIM_Y, "try-error")) {return(list(IC1 = NA, IC2 = NA))}

    intersection_function <- function(x) {
      dnorm(x, mu1_hat, sigma1_hat) - dnorm_mixture(x, mu2_hat, sigma2_hat, pi2_hat)
    }

    intersec <- encontrar_raices(intersection_function, interv)
    OVL_hat <- OVL_mix(mu1_hat, mu2_hat, sigma1_hat, sigma2_hat, 1, pi2_hat, intersec)

    OVL_deriv <- OVL_derivates_1(mu1_hat=mu1_hat, mu2_hat=mu2_hat, sigma1_hat=sigma1_hat,
                                 sigma2_hat=sigma2_hat, pi2_hat=pi2_hat, h=h,
                                 intersec=intersec, OVL_mix)
    deriv1 <- OVL_deriv$deriv1
    deriv2 <- OVL_deriv$deriv2
    deriv3 <- OVL_deriv$deriv3
    deriv4 <- OVL_deriv$deriv4
    deriv5 <- OVL_deriv$deriv5
    deriv6 <- OVL_deriv$deriv6
    deriv7 <- OVL_deriv$deriv7

    I11 <- n1/(sigma1_hat)^2
    I22 <- (2*n1^2)/((n1 - 1) * (sigma1_hat)^2)

    intervalo <- try({
      I_X <- diag(c(I11, I22))
      I <- as.matrix(bdiag(I_X, FIM_Y))
      cov <- solve(I)
      vector_deriv <- t(as.matrix(c(deriv1, deriv2, deriv3, deriv4, deriv5, deriv6, deriv7)))
      vector_deriv_t <- t(vector_deriv)
      var_OVL <- vector_deriv %*% cov %*% vector_deriv_t

      IC1 <- OVL_hat - qnorm(1 - alpha/2) * sqrt(var_OVL)
      IC1_aux <- IC1
      if(IC1 < 0 & (is.na(IC1) == FALSE)){IC1 <- 0}else{IC1 <- IC1_aux}

      IC2 <- OVL_hat + qnorm(1 - alpha/2) * sqrt(var_OVL)
      IC2_aux <- IC2
      if(IC2 > 1 & (is.na(IC2) == FALSE)){IC2 <- 1}else{IC2 <- IC2_aux}

      return(list(IC1 = IC1, IC2 = IC2, var_OVL = var_OVL,
                  pi2_hat = pi2_hat, OVL_hat = OVL_hat))
    }, silent = TRUE)

    if (inherits(intervalo, "try-error")) {
      return(list(IC1 = NA, IC2 = NA, var_OVL = NA, pi2_hat = pi2_hat, OVL_hat = OVL_hat))
    } else {
      return(intervalo)
    }

  } else {

    n1 <- length(x)
    n2 <- length(y)

    EM_result_x <- try(normalmixEM(x), silent = TRUE)
    if (inherits(EM_result_x, "try-error")){
      EM_result_x <- EM(x)
      mu1_hat <- EM_result_x$mu
      sigma1_hat <- EM_result_x$sigma
      pi1_hat <- EM_result_x$pi
    } else {
      mu1_hat <- EM_result_x$mu
      sigma1_hat <- EM_result_x$sigma
      pi1_hat <- EM_result_x$lambda
    }

    EM_result_y <- try(normalmixEM(y), silent = TRUE)
    if (inherits(EM_result_y, "try-error")){
      EM_result_y <- EM(y)
      mu2_hat <- EM_result_y$mu
      sigma2_hat <- EM_result_y$sigma
      pi2_hat <- EM_result_y$pi
    } else {
      mu2_hat <- EM_result_y$mu
      sigma2_hat <- EM_result_y$sigma
      pi2_hat <- EM_result_y$lambda
    }

    if(any(is.na(c(c(mu1_hat, sigma1_hat, pi1_hat), c(mu2_hat, sigma2_hat, pi2_hat)))) == TRUE){
      return(list(IC1 = NA, IC2 = NA, pi1_hat = NA, pi2_hat = NA, OVL_hat = NA))
    }

    initial_params_x <- list(pi = pi1_hat[1], mu1 = mu1_hat[1], mu2 = mu1_hat[2],
                             sigma1 = sigma1_hat[1], sigma2 = sigma1_hat[2])

    FIM_X <- try(FIM_mixture_normals(data = x, params = initial_params_x), silent = TRUE)
    if (inherits(FIM_X, "try-error")){return(list(IC1 = NA, IC2 = NA))}

    initial_params_y <- list(pi = pi2_hat[1], mu1 = mu2_hat[1], mu2 = mu2_hat[2],
                             sigma1 = sigma2_hat[1], sigma2 = sigma2_hat[2])

    FIM_Y <- try(FIM_mixture_normals(data = y, params = initial_params_y), silent = TRUE)
    if (inherits(FIM_Y, "try-error")){return(list(IC1 = NA, IC2 = NA))}

    intersection_function <- function(x) {
      dnorm_mixture(x, mu1_hat, sigma1_hat, pi1_hat) - dnorm_mixture(x, mu2_hat, sigma2_hat, pi2_hat)
    }

    intersec <- encontrar_raices(intersection_function, interv)
    OVL_hat <- OVL_mix(mu1_hat, mu2_hat, sigma1_hat, sigma2_hat, pi1_hat, pi2_hat, intersec)

    OVL_deriv <- OVL_derivates_2(mu1_hat=mu1_hat, mu2_hat=mu2_hat, sigma1_hat=sigma1_hat,
                                 sigma2_hat=sigma2_hat, pi1_hat=pi1_hat, pi2_hat=pi2_hat,
                                 h=h, intersec=intersec, OVL_mix)
    deriv1 <- OVL_deriv$deriv1
    deriv2 <- OVL_deriv$deriv2
    deriv3 <- OVL_deriv$deriv3
    deriv4 <- OVL_deriv$deriv4
    deriv5 <- OVL_deriv$deriv5
    deriv6 <- OVL_deriv$deriv6
    deriv7 <- OVL_deriv$deriv7
    deriv8 <- OVL_deriv$deriv8
    deriv9 <- OVL_deriv$deriv9
    deriv10 <- OVL_deriv$deriv10

    intervalo <- try({
      I <- as.matrix(bdiag(FIM_X, FIM_Y))
      cov <- solve(I)
      vector_deriv <- t(as.matrix(c(deriv1, deriv2, deriv3, deriv4, deriv5,
                                    deriv6, deriv7, deriv8, deriv9, deriv10)))
      vector_deriv_t <- t(vector_deriv)
      var_OVL <- vector_deriv %*% cov %*% vector_deriv_t

      IC1 <- OVL_hat - qnorm(1 - alpha/2) * sqrt(var_OVL)
      IC1_aux <- IC1
      if(IC1 < 0 & (is.na(IC1) == FALSE)){IC1 <- 0}else{IC1 <- IC1_aux}

      IC2 <- OVL_hat + qnorm(1 - alpha/2) * sqrt(var_OVL)
      IC2_aux <- IC2
      if(IC2 > 1 & (is.na(IC2) == FALSE)){IC2 <- 1}else{IC2 <- IC2_aux}

      return(list(IC1 = IC1, IC2 = IC2, var_OVL = var_OVL,
                  pi1_hat = pi1_hat, pi2_hat = pi2_hat, OVL_hat = OVL_hat))
    }, silent = TRUE)

    if (inherits(intervalo, "try-error")) {
      return(list(IC1 = NA, IC2 = NA, var_OVL = NA,
                  pi1_hat = pi1_hat, pi2_hat = pi2_hat, OVL_hat = OVL_hat))
    } else {
      return(intervalo)
    }
  }
}

