#' LDA projection pursuit index
#' 
#' Calculate LDA projection pursuit index value
#' @title LDA PPindex
#' @usage LDAindex(origclass,origdata,proj,weight=TRUE)
#' @param origclass class information vector
#' @param origdata data matrix without class information  
#' @param proj projection matrix 
#' @param weight weight flag in LDA
#' @references Lee, EK., Cook, D., Klinke, S., and Lumley, T.(2005) 
#' Projection Pursuit for exploratory supervised classification, 
#' Journal of Computational and Graphical statistics, 14(4):831-846.
#' @export
#' @import Rcpp
#' @keywords projection pursuit
#' @examples
#' data(iris)
#' LDAindex(iris[,5],as.matrix(iris[,1:4]))

LDAindex <- function(origclass, origdata, proj = matrix(0), weight = TRUE) {
  .LDAindex(factor(origclass), origdata, proj, weight)
}
#' PDA projection pursuit index 
#' 
#' Calculate PDA projection pursuit index value 
#' @title PDA PPindex
#' @usage PDAindex(origclass,origdata,proj,weight=TRUE,lambda=0.1)
#' @param origclass class information vector
#' @param origdata data matrix without class information  
#' @param proj projection matrix 
#' @param weight weight flag in PDA
#' @param lambda lambda in PDA index
#' @references Lee, EK., Cook, D.(2010) 
#' A projection pursuit index for large p small n data, 
#' Statistics and Computing, 20:381-392.
#' @export
#' @keywords projection pursuit
#' @examples
#' data(iris)
#' PDAindex(iris[,5],as.matrix(iris[,1:4]),lambda=0.2)
#' 
#' 
PDAindex <- function(origclass, origdata, proj = matrix(0), weight = TRUE, lambda = 0.1) {
  .PDAindex(factor(origclass), origdata, proj, weight)
}

#' Lr projection pursuit index
#' 
#' Calculate Lr projection pursuit index value
#' @title Lr PPindex
#' @usage Lrindex(origclass,origdata,proj,weight=TRUE,r=1)
#' @param origclass class information vector
#' @param origdata data matrix without class information  
#' @param proj projection matrix 
#' @param weight weight flag in Lr index
#' @param r r in Lr index
#' @references Lee, EK., Cook, D., Klinke, S., and Lumley, T.(2005) 
#' Projection Pursuit for exploratory supervised classification, 
#' Journal of Computational and Graphical statistics, 14(4):831-846.
#' @export
#' @keywords projection pursuit
#' @examples
#' data(iris)
#' Lrindex(iris[,5],as.matrix(iris[,1:4]),r=1)
#' 
#'   
Lrindex <- function(origclass, origdata, proj = matrix(0), weight = TRUE, r = 1L) {
  .Lrindex(factor(origclass), origdata, proj, weight, r)
}

#' GINI projection pursuit index
#' 
#' Calculate GINI projection pursuit index value
#' @title GINI PPindex
#' @usage GINIindex1D(origclass,origdata,proj)
#' @param origclass class information vector
#' @param origdata data matrix without class information  
#' @param proj projection matrix
#' @export
#' @keywords projection pursuit
#' @examples
#' data(iris)
#' GINIindex1D(iris[,5],as.matrix(iris[,1,drop=FALSE]))
#' 
GINIindex1D <- function(origclass, origdata, proj = numeric(0)) {
  .GINIindex1D(factor(origclass), origdata, proj)
}

#' ENTROPY projection pursuit index 
#' 
#' Calculate ENTROPY projection pursuit index value
#' @title ENTROPY PPindex
#' @usage ENTROPYindex1D(origclass,origdata,proj)
#' @param origclass class information vector
#' @param origdata data matrix without class information  
#' @param proj projection matrix
#' @export
#' @keywords projection pursuit
#' @examples
#' data(iris)
#' ENTROPYindex1D(iris[,5],as.matrix(iris[,1,drop=FALSE]))
ENTROPYindex1D <- function(origclass, origdata, proj = numeric(0)) {
  .ENTROPYindex1D(factor(origclass), origdata, proj)
}
#' PP optimization using various projection pursuit indices
#' 
#' Find the q-dim optimal projection using various projectin pursuit indices 
#' with class information
#' @title Projection pursuit optimization
#' @usage PPopt(origclass,origdata,q=1,PPmethod="LDA",weight=TRUE,r=1,
#'              lambda=0.1,energy=0,cooling=0.999,TOL=0.0001,maxiter = 50000)
#' @param origclass class information vector
#' @param origdata data matrix without class information
#' @param q dimension of projection matrix
#' @param PPmethod method for projection pursuit; "LDA", "PDA", "Lr", "GINI", 
#'                 and "ENTROPY"
#' @param weight weight flag in LDA, PDA and Lr index
#' @param r r in Lr index
#' @param lambda lambda in PDA index
#' @param energy energy parameter
#' @param cooling cooling parameter
#' @param TOL tolerance
#' @param maxiter number of maximum iteration
#' @return indexbest maximum LDA index value
#' @return projbest optimal q-dim projection matrix
#' @return origclass original class information vector
#' @return origdata  original data matrix  without class information
#' @references Lee, EK., Cook, D., Klinke, S., and Lumley, T.(2005) 
#' Projection Pursuit for exploratory supervised classification, 
#' Journal of Computational and Graphical statistics, 14(4):831-846.
#' @export
#' @keywords projection pursuit
#' @examples
#' data(iris)
#' PP.proj.result <- PPopt(iris[,5],as.matrix(iris[,1:4]))
#' PP.proj.result
PPopt <- function(origclass, origdata, q = 1L, PPmethod = "LDA", weight = TRUE, r = 1L, lambda = 0.1, energy = 0, cooling = 0.999, TOL = 0.0001, maxiter = 50000L) {
  result<-.PPopt(factor(origclass), origdata, q, PPmethod, weight, r, lambda, energy, cooling, TOL, maxiter)
  class(result)<-append(class(result),"PPoptim")
  return(result)
}