\name{compute_psri_sm}
\alias{compute_psri_sm}
\title{Compute Softmax Plant Stress Response Index (PSRI_SM)}
\description{
Calculates the softmax-aggregated PSRI from time-series germination data.
This method handles zero-value components gracefully through adaptive
softmax reweighting, eliminating the zero-collapse failure mode of the
geometric mean approach.
}
\usage{
compute_psri_sm(
  germination_counts,
  timepoints,
  total_seeds,
  radicle_count = NULL,
  temperature = 0.13,
  return_components = FALSE
)
}
\arguments{
  \item{germination_counts}{Integer vector of cumulative germination counts
    at each timepoint. Must be non-decreasing and not exceed
    \code{total_seeds}.}
  \item{timepoints}{Numeric vector of observation times (e.g., days).
    Must be the same length as \code{germination_counts} and strictly
    increasing.}
  \item{total_seeds}{Integer. Total number of seeds in the replicate.}
  \item{radicle_count}{Integer or \code{NULL}. Number of seeds with
    radicle emergence at the final observation. If \code{NULL}, radicle
    vigor is excluded (3-component PSRI). Default is \code{NULL}.}
  \item{temperature}{Numeric. Softmax temperature parameter. Default is
    0.13, calibrated via perplexity targeting. See
    \code{\link{calibrate_temperature}}.}
  \item{return_components}{Logical. If \code{TRUE}, returns a list
    containing the PSRI value, all component scores, and softmax weights.
    Default is \code{FALSE}.}
}
\value{
  If \code{return_components = FALSE} (default), a single numeric
  PSRI value.

  If \code{return_components = TRUE}, a list with:
  \describe{
    \item{psri_sm}{The softmax PSRI value.}
    \item{components}{Named numeric vector of component scores.}
    \item{weights}{Named numeric vector of softmax weights.}
    \item{temperature}{The temperature parameter used.}
    \item{n_components}{Number of components used (3 or 4).}
  }
}
\details{
The softmax PSRI aggregates germination components using weighted
summation:
\deqn{PSRI_{SM} = \sum W_i \cdot C_i}
where \eqn{C = [MSG, MRG, cMTG]} (3-component) or
\eqn{C = [MSG, MRG, cMTG, RVS]} (4-component), and weights are:
\deqn{W_i = \frac{\exp(C_i / T)}{\sum_j \exp(C_j / T)}}

When any component \eqn{C_k = 0}, its softmax weight approaches zero
and the remaining components absorb its weight, preserving information
rather than collapsing the index.
}
\examples{
# Basic 3-component PSRI
compute_psri_sm(
  germination_counts = c(5, 15, 20),
  timepoints = c(3, 5, 7),
  total_seeds = 25
)

# With radicle vigor (4-component)
compute_psri_sm(
  germination_counts = c(5, 15, 20),
  timepoints = c(3, 5, 7),
  total_seeds = 25,
  radicle_count = 18
)

# Detailed output
result <- compute_psri_sm(
  germination_counts = c(0, 0, 10),
  timepoints = c(3, 5, 7),
  total_seeds = 25,
  return_components = TRUE
)
result$psri_sm     # > 0 (softmax prevents collapse)
result$components
result$weights
}
\seealso{
\code{\link{compute_msg}}, \code{\link{compute_mrg}},
\code{\link{compute_cmtg}}, \code{\link{compute_rvs}},
\code{\link{softmax_weights}}, \code{\link{calibrate_temperature}}
}
\references{
Walne, C.H., Gaudin, A., Henry, W.B., and Reddy, K.R. (2020).
In vitro seed germination response of corn hybrids to osmotic stress
conditions. \emph{Agrosystems, Geosciences & Environment}, 3(1), e20087.
\doi{10.1002/agg2.20087}
}
