#' Read and merge spectral data using data.table exclusively
#' @param directory Path to the directory containing files
#' @param pattern File pattern to search for (e.g., 'input')
#' @param output_path Path to save the processed output
#' @return A merged data.table
#' @export
#' @examples
#' library(data.table)
#' # Create mock data files with all required columns
#' mock_data1 <- data.frame(
#'   treatment = sample(0:1, 50, replace = TRUE),
#'   var1 = rnorm(50),
#'   var2 = rnorm(50),
#'   Scan.date = sample(
#'     seq.Date(
#'       from = as.Date('2023-01-01'),
#'       to = as.Date('2023-12-31'),
#'       by = 'day'
#'     ),
#'     50,
#'     replace = TRUE
#'   ),
#'   Scan.time = format(Sys.time(), "%H:%M:%S"),
#'   timestamp = Sys.time()  # Add timestamp column
#' )
#'
#' mock_data2 <- data.frame(
#'   treatment = sample(0:1, 50, replace = TRUE),
#'   var1 = rnorm(50),
#'   var2 = rnorm(50),
#'   Scan.date = sample(
#'     seq.Date(
#'       from = as.Date('2023-01-01'),
#'       to = as.Date('2023-12-31'),
#'       by = 'day'
#'     ),
#'     50,
#'     replace = TRUE
#'   ),
#'   Scan.time = format(Sys.time(), "%H:%M:%S"),
#'   timestamp = Sys.time()  # Add timestamp column
#' )
#'
#' # Save mock data to temporary CSV files
#' temp_dir <- tempdir()
#' file1 <- file.path(temp_dir, "input_file1.csv")
#' file2 <- file.path(temp_dir, "input_file2.csv")
#' fwrite(mock_data1, file1)
#' fwrite(mock_data2, file2)
#'
#' # Run the reads() function on mock CSV data
#' merged_data <- reads(
#'   directory = temp_dir,
#'   pattern = "input",
#'   output_path = tempfile(fileext = ".csv")
#' )
#' head(merged_data)
reads <- function(directory, pattern, output_path) {
  
  # Step 1: List all matching files in the directory
  file_paths <- list.files(path = directory, pattern = pattern, full.names = TRUE)
  
  # Step 2: Read all files and ensure consistent column types
  data_list <- lapply(file_paths, function(file) {
    data <- data.table::fread(file)
    
    # Automatically detect variations of 'treatment' column
    treatment_col <- grep("treatment", colnames(data), ignore.case = TRUE, value = TRUE)
    
    # If 'treatment' column is not found, skip the file
    if (length(treatment_col) == 0) {
      warning(paste("The 'treatment' column is missing in file:", file, "- Skipping file"))
      return(NULL)
    }
    
    # Standardize the column name
    colnames(data)[colnames(data) == treatment_col] <- "treatment"
    
    # Ensure the treatment column is always character
    data$treatment <- as.character(data$treatment)
    
    # Ensure the timestamp column is always POSIXct
    if ("timestamp" %in% colnames(data)) {
      data$timestamp <- lubridate::ymd_hms(data$timestamp)  # More robust format
    }
    
    return(data)
  })
  
  # Step 3: Remove NULL entries (files without treatment column)
  data_list <- Filter(Negate(is.null), data_list)
  
  # Step 4: Merge all datasets
  merged_data <- data.table::rbindlist(data_list, fill = TRUE)
  
  # Step 5: Remove rows where treatment is 'blanc'
  merged_data <- merged_data[treatment != "blanc", ]
  
  # Step 6: Split the timestamp column into Scan.date and Scan.time
  if ("timestamp" %in% colnames(merged_data)) {
    merged_data <- dplyr::mutate(
      merged_data,
      Scan.date = format(timestamp, "%d-%m-%Y"),
      Scan.time = format(timestamp, "%H:%M:%S")
    )
  }
  
  # Step 7: Create a Sample column based on the treatment group
  merged_data <- merged_data %>%
    dplyr::group_by(treatment, Scan.date, Scan.time) %>%
    dplyr::mutate(Sample = dplyr::row_number()) %>%
    dplyr::ungroup() %>%
    dplyr::select(treatment, Sample, Scan.date, Scan.time, dplyr::everything(), -timestamp)
  
  # Step 8: Export the merged dataset as an Excel file
  openxlsx::write.xlsx(merged_data, output_path)
  
  # Return the merged and processed dataset
  return(merged_data)
}
