#' Visualize Plate Layout (Heatmap)
#'
#' Creates a physical map of the 96-well plate colored by Ct value or Sample.
#'
#' @param data The dataframe output from import_plate().
#' @param fill_var The column to color the wells by (e.g., "Ct", "Sample", "Gene").
#' @param interactive If TRUE, returns a plotly interactive graph. If FALSE, returns static ggplot.
#' @return A ggplot object or plotly object.
#' @export
#' @examples
#' # Create dummy data representing a partial plate
#' dummy_data <- data.frame(
#'   Well = c("A1", "A2", "A3", "B1", "B2", "B3"),
#'   Ct = c(20, 20.5, 19.8, 25, 24.5, 26),
#'   Sample = c(rep("Control", 3), rep("Treated", 3)),
#'   Gene = "GAPDH"
#' )
#'
#' # View static heatmap
#' view_plate(dummy_data, fill_var = "Ct")
#'
#' # View interactive heatmap (if library plotly is available)
#' view_plate(dummy_data, fill_var = "Sample", interactive = FALSE)
view_plate <- function(data, fill_var = "Ct", interactive = FALSE) {

  # 1. Parse Well Position
  plot_data <- data %>%
    dplyr::mutate(
      Row = stringr::str_sub(Well, 1, 1),
      Col = as.numeric(stringr::str_sub(Well, 2)),
      Tooltip = paste0("Well: ", Well, "\nSample: ", Sample, "\nGene: ", Gene, "\nCt: ", round(Ct, 2))
    )

  # 2. Build Plot
  p <- ggplot2::ggplot(plot_data, ggplot2::aes(x = Col, y = dplyr::desc(Row), text = Tooltip)) +
    ggplot2::geom_tile(ggplot2::aes(fill = .data[[fill_var]]), color = "white", size = 0.5) +
    ggplot2::geom_text(ggplot2::aes(label = round(Ct, 1)), size = 2.5, color = "black") +
    ggplot2::scale_x_continuous(breaks = 1:12, position = "top") +
    ggplot2::labs(title = paste("PlateVision QC:", fill_var), x = NULL, y = NULL) +
    ggplot2::theme_minimal() +
    ggplot2::theme(panel.grid = ggplot2::element_blank())

  # 3. Coloring Logic
  if(is.numeric(plot_data[[fill_var]])) {
    p <- p + ggplot2::scale_fill_viridis_c(option = "magma", direction = -1, name = "Ct Value")
  } else {
    p <- p + ggplot2::scale_fill_brewer(palette = "Set3", name = fill_var)
  }

  if(interactive) {
    return(plotly::ggplotly(p, tooltip = "text"))
  } else {
    return(p)
  }
}
