% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IPWE_mean_IndCen.R
\name{IPWE_mean_IndCen}
\alias{IPWE_mean_IndCen}
\title{Estimate the mean-optimal treatment regime for data with independently censored response}
\usage{
IPWE_mean_IndCen(data, regimeClass, moPropen = "BinaryRandom",
  Domains = NULL, cluster = FALSE, p_level = 1, s.tol = 1e-04,
  it.num = 8, pop.size = 3000)
}
\arguments{
\item{data}{a data.frame, containing variables in the \code{moPropen} and 
\code{RegimeClass} and also the response variables, namely \code{censor_y} as the censored response, 
and \code{delta} as the censoring indicator.}

\item{regimeClass}{a formula specifying the class of treatment regimes to search,
e.g. if \code{regimeClass = a~x1+x2}, and then this function will 
search the class of treatment regimes
of the form 
\deqn{d(x) = I \left(\beta_0 +\beta_1  x_1 + \beta_2  x_2 > 0\right).
}{d(x)=I(\beta_0 +\beta_1 * x1  + \beta_2 * x2 > 0).}
Polynomial arguments are also supported.}

\item{moPropen}{The propensity score model for the probability of receiving 
treatment level 1.
When \code{moPropen} equals the string "BinaryRandom",  the proportion of observations
receiving treatment level 1 in the sample will be plugged in as an estimate
of the propensity.
Otherwise, this argument should be a formula/string, based on which this function
will fit a logistic regression on the treatment level.  e.g. \code{a1~x1}.}

\item{Domains}{default is NULL. Otherwise, the object should be a \code{nvars *2} 
matrix used as the space of parameters, which will be supplied to \code{rgenoud::genoud}. 
\code{nvars} is the total number of parameters.}

\item{cluster}{default is FALSE, meaning do not use parallel computing for the genetic algorithm(GA).}

\item{p_level}{choose between 0,1,2,3 to indicate different levels of output
from the genetic function. Specifically, 0 (minimal printing),
  1 (normal), 2 (detailed), and 3 (debug).}

\item{s.tol}{tolerance level for the GA algorithm. This is input for parameter \code{solution.tolerance}
in function \code{rgenoud::genoud}.}

\item{it.num}{the maximum GA iteration number}

\item{pop.size}{an integer with the default set to be 3000. This is roughly the 
number individuals for the first generation
in the genetic algorithm (\code{rgenoud::genoud}).}
}
\value{
This function returns an object with 6 objects: 
\itemize{
 \item{\code{coefficients}}{ the estimated parameter indexing the mean-optimal treatment regime. 
         Since we focus the space of linear treatment regimes, the estimated decision rule
         cannot be uniquely identified without scale normalized. In this package,
         we normalized by \eqn{|\beta_1| = 1}, which was proposed in Horowitz \insertCite{horowitz1992smoothed}{QTOCen}.     }
 \item{\code{hatQ}} { the estimated optimal marginal mean response} 
 \item{\code{moPropen}}{ log of the input argument of \code{moPropen}}
 \item{\code{regimeClass}}{ log of the input argument of \code{regimeClass}}
 \item{\code{data_aug}}{ Training data with additional columns used in the algorithm. Note that \code{data_aug} is used for plotting 
 of survival function of the censoring time}
 \item{\code{survfitCensorTime}}{ the estimated survival function of the censoring time}
}
}
\description{
This function estimates the Mean-optimal Treatment Regime 
with censored response. 
The implemented function only works for scenarios in which 
treatment is binary and the censoring time
is independent of baseline covariates, treatment group and all potential survival times.
}
\examples{
GenerateData <- function(n)
{
  x1 <- runif(n, min=-0.5,max=0.5)
  x2 <- runif(n, min=-0.5,max=0.5)
  error <- rnorm(n, sd= 1)
  ph <- exp(-0.5+1*(x1+x2))/(1+exp(-0.5 + 1*(x1+x2)))
  a <- rbinom(n = n, size = 1, prob=ph)
  c <- 1.5 +  + runif(n = n, min=0, max=2)
  cmplt_y <-  pmin(2+x1+x2 +  a*(1 - x1 - x2) +  (0.2 + a*(1+x1+x2)) * error, 4.4)
  censor_y <- pmin(cmplt_y, c)
  delta <- as.numeric(c > cmplt_y)
  return(data.frame(x1=x1,x2=x2,a=a, censor_y = censor_y, delta=delta))
}
n <- 400

D <- GenerateData(n)
fit1 <- IPWE_mean_IndCen(data = D, regimeClass = a~x1+x2)
                                 
   
}
\references{
\insertRef{zhou2018quantile}{QTOCen}

\insertRef{horowitz1992smoothed}{QTOCen}
}
