% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/S3Extract.R
\name{extract}
\alias{extract}
\alias{extract.CDM}
\alias{extract.validation}
\alias{extract.sim.data}
\alias{extract.fit}
\alias{extract.is.Qident}
\alias{extract.att.hierarchy}
\title{Extract Components from Qval Package Objects}
\usage{
extract(object, what, ...)

\method{extract}{CDM}(object, what, ...)

\method{extract}{validation}(object, what, ...)

\method{extract}{sim.data}(object, what, ...)

\method{extract}{fit}(object, what, ...)

\method{extract}{is.Qident}(object, what, ...)

\method{extract}{att.hierarchy}(object, what, ...)
}
\arguments{
\item{object}{An object of class \code{\link[Qval]{CDM}}, \code{\link[Qval]{validation}}, 
\code{\link[Qval]{sim.data}}, \code{\link[Qval]{fit}}, \code{\link[Qval]{is.Qident}}, 
\code{\link[Qval]{att.hierarchy}}.}

\item{what}{A character string specifying the name of the component to extract.}

\item{...}{Additional arguments (currently ignored).}
}
\value{
The requested component. The return type depends on the specified \code{what} and the class of the \code{object}.
}
\description{
A unified extractor function for retrieving internal components from objects produced by the \pkg{Qval} package. 
This method allows users to access key elements such as model results, validation logs, and simulation settings 
in a structured and object-oriented manner.
}
\details{
This generic extractor supports three core object classes: \code{\link[Qval]{CDM}}, \code{\link[Qval]{validation}}, 
\code{\link[Qval]{sim.data}}, \code{\link[Qval]{fit}}, \code{\link[Qval]{is.Qident}}, \code{\link[Qval]{att.hierarchy}}. 
It is intended to streamline access to commonly used internal components without manually referencing object slots.
The available components for each class are listed below:

\describe{
  \item{\strong{\code{CDM}}}{Cognitive Diagnosis Model fitting results. Available components:
    \describe{
      \item{\code{analysis.obj}}{The internal model fitting object (e.g., GDINA or Baseline Model).}
      \item{\code{alpha}}{Estimated attribute profiles (EAP estimates) for each respondent.}
      \item{\code{P.alpha.Xi}}{Posterior distribution of latent attribute patterns.}
      \item{\code{alpha.P}}{Marginal attribute mastery probabilities (estimated).}
      \item{\code{P.alpha}}{Prior attribute probabilities at convergence.}
      \item{\code{pattern}}{The attribute mastery pattern matrix containing all possible attribute mastery pattern.}
      \item{\code{Deviance}}{Negative twice the marginal log-likelihood (model deviance).}
      \item{\code{npar}}{Number of free parameters estimated in the model.}
      \item{\code{AIC}}{Akaike Information Criterion.}
      \item{\code{BIC}}{Bayesian Information Criterion.}
      \item{\code{call}}{The original model-fitting function call.}
      \item{\code{...}}{Can \code{\link[GDINA]{extract}} corresponding value from the \code{\link[GDINA]{GDINA}} object.}
    }}

  \item{\strong{\code{validation}}}{Q-matrix validation results. Available components:
    \describe{
      \item{\code{Q.orig}}{The original Q-matrix submitted for validation.}
      \item{\code{Q.sug}}{The suggested (revised) Q-matrix after validation.}
      \item{\code{time.cost}}{Total computation time for the validation procedure.}
      \item{\code{process}}{Log of Q-matrix modifications across iterations.}
      \item{\code{iter}}{Number of iterations performed during validation.}
      \item{\code{priority}}{Attribute priority matrix (available for PAA-based methods only).}
      \item{\code{Hull.fit}}{Data required to plot the Hull method results (for Hull-based validation only).}
      \item{\code{call}}{The original function call used for validation.}
    }}

  \item{\strong{\code{sim.data}}}{Simulated data and parameters used in cognitive diagnosis simulation studies:
    \describe{
      \item{\code{dat}}{Simulated dichotomous response matrix (\eqn{N \times I}).}
      \item{\code{Q}}{Q-matrix used to generate the data.}
      \item{\code{attribute}}{True latent attribute profiles (\eqn{N \times K}).}
      \item{\code{catprob.parm}}{Item-category conditional success probabilities (list format).}
      \item{\code{delta.parm}}{Item-level delta parameters (list format).}
      \item{\code{higher.order.parm}}{Higher-order model parameters (if used).}
      \item{\code{mvnorm.parm}}{Parameters for the multivariate normal attribute distribution (if used).}
      \item{\code{LCprob.parm}}{Latent class-based success probability matrix.}
      \item{\code{call}}{The original function call that generated the simulated data.}
    }}
    
  \item{\strong{\code{fit}}}{Relative fit indices (-2LL, AIC, BIC, CAIC, SABIC) and absolute fit indices (\eqn{M_2} test, \eqn{RMSEA_2}, SRMSR):
    \describe{
       \item{\code{npar}}{The number of parameters.}
       \item{\code{-2LL}}{The Deviance.}
       \item{\code{AIC}}{The Akaike information criterion.}
       \item{\code{BIC}}{The Bayesian information criterion.}
       \item{\code{CAIC}}{The consistent Akaike information criterion.}
       \item{\code{SABIC}}{The Sample-size Adjusted BIC.}
       \item{\code{M2}}{A vector consisting of \eqn{M_2} statistic, degrees of freedom, significance level, and \eqn{RMSEA_2} (Liu, Tian, & Xin, 2016).}
       \item{\code{SRMSR}}{The standardized root mean squared residual (SRMSR; Ravand & Robitzsch, 2018).}
     }
   }
   
   \item{\strong{\code{is.Qident}}}{Results of whether the Q-matrix is identifiable:
     \describe{
       \item{\code{completeness}}{TRUE if \eqn{K \times K} identity submatrix exists.}
       \item{\code{distinctness}}{TRUE if remaining columns are distinct.}
       \item{\code{repetition}}{TRUE if every attribute appears more than 3 items.}
       \item{\code{genericCompleteness}}{TRUE if two different generic complete \eqn{K \times K} submatrices exist.}
       \item{\code{genericRepetition}}{TRUE if at least one '1' exists outside those submatrices.}
       \item{\code{Q1, Q2}}{Identified generic complete submatrices (if found).}
       \item{\code{Q.star}}{Remaining part after removing rows in \code{Q1} and \code{Q2}.}
       \item{\code{locallyGenericIdentifiability}}{TRUE if local generic identifiability holds.}
       \item{\code{globallyGenericIdentifiability}}{TRUE if global generic identifiability holds.}
       \item{\code{Q.reconstructed.DINA}}{Reconstructed Q-matrix with low-frequency attribute moved to first column.}
     }
   }
   
   \item{\strong{\code{att.hierarchy}}}{Results of iterative attribute hierarchy exploration:
     \describe{
       \item{\code{noSig}}{TRUE all structural parameters are not greater than 0.}
       \item{\code{isNonverge}}{TRUE if convergence was achieved.}
       \item{\code{statistic}}{A 4-column \code{data.frame} results for each structural parameter that is significantly larger than 0.}
       \item{\code{pattern}}{The attribute pattern matrix under iterative attribute hierarchy.}
     }
   }
 }
}
\section{Methods (by class)}{
\itemize{
\item \code{extract(CDM)}: Extract fields from a \code{CDM} object

\item \code{extract(validation)}: Extract fields from a \code{validation} object

\item \code{extract(sim.data)}: Extract fields from a \code{sim.data} object

\item \code{extract(fit)}: Extract fields from a \code{fit} object

\item \code{extract(is.Qident)}: Extract fields from a \code{is.Qident} object

\item \code{extract(att.hierarchy)}: Extract fields from a \code{att.hierarchy} object

}}
\examples{
library(Qval)
set.seed(123)

\donttest{
################################################################
# Example 1: sim.data extraction                               #
################################################################
Q <- sim.Q(3, 10)
data.obj <- sim.data(Q, N = 200)
extract(data.obj, "dat")


################################################################
# Example 2: CDM extraction                                    #
################################################################
CDM.obj <- CDM(data.obj$dat, Q)
extract(CDM.obj, "alpha")
extract(CDM.obj, "AIC")


################################################################
# Example 3: validation extraction                             #
################################################################
validation.obj <- validation(data.obj$dat, Q, CDM.obj)
Q.sug <- extract(validation.obj, "Q.sug")
print(Q.sug)


################################################################
# Example 4: fit extraction                                    #
################################################################
fit.obj <- fit(data.obj$dat, Q.sug, model="GDINA")
extract(fit.obj, "M2")
}

}
\references{
Khaldi, R., Chiheb, R., & Afa, A.E. (2018). Feed-forward and Recurrent Neural Networks for Time Series Forecasting: Comparative Study. In: Proceedings of the International Conference on Learning and Optimization Algorithms: Theory and Applications (LOPAL 18). Association for Computing Machinery, New York, NY, USA, Article 18, 1–6. DOI: 10.1145/3230905.3230946.

Liu, Y., Tian, W., & Xin, T. (2016). An application of M2 statistic to evaluate the fit of cognitive diagnostic models. Journal of Educational and Behavioral Statistics, 41, 3–26. DOI: 10.3102/1076998615621293.

Ravand, H., & Robitzsch, A. (2018). Cognitive diagnostic model of best choice: a study of reading comprehension. Educational Psychology, 38, 1255–1277. DOI: 10.1080/01443410.2018.1489524.
}
