#' Bayesian Changepoint Detection Methods
#'
#' @description
#' Implementation of Bayesian methods for changepoint detection:
#' BOCPD (Bayesian Online Changepoint Detection) and Shiryaev-Roberts.
#'
#' @name bayesian_methods
#' @noRd
NULL

#' Bayesian Online Changepoint Detection
#'
#' @description
#' Implements the BOCPD algorithm of Adams and MacKay (2007).
#' Maintains a posterior distribution over the run length (time since
#' last changepoint) and updates it online as new observations arrive.
#'
#' @param data Numeric vector or matrix
#' @param type Type of change to detect (currently supports "both")
#' @param prior Prior specification (from normal_gamma(), normal_known_var(), etc.)
#' @param hazard Hazard prior (from geometric_hazard(), etc.) or numeric hazard rate
#' @param threshold Probability threshold for declaring a changepoint
#' @param truncate_run_length Maximum run length to track (for efficiency)
#' @param ... Additional arguments
#'
#' @return List containing:
#'   \itemize{
#'     \item changepoints: Detected changepoint locations
#'     \item posterior: Matrix of run length posteriors over time
#'     \item prob_change: Probability of changepoint at each time
#'     \item map_run_length: Maximum a posteriori run length at each time
#'   }
#'
#' @references
#' Adams, R. P. and MacKay, D. J. C. (2007). Bayesian Online Changepoint Detection.
#' arXiv:0710.3742
#'
#' @examples
#' data <- c(rnorm(100, mean = 0), rnorm(100, mean = 3))
#'
#' result <- bocpd(data)
#'
#' result <- bocpd(data, prior = normal_gamma(mu0 = 0, kappa0 = 1,
#'                                             alpha0 = 1, beta0 = 1))
#'
#'
#' result <- bocpd(data, hazard = 0.01)
#'
#' @export
bocpd <- function(data, type = "both", prior = NULL, hazard = NULL,
                  threshold = 0.3, truncate_run_length = NULL, ...) {
  detect_bocpd(data, type, prior, hazard, threshold, truncate_run_length, ...)
}

#' Extract hazard rate from hazard specification
#' @noRd
get_hazard_rate <- function(hazard) {
  if (is.numeric(hazard)) {
    return(hazard)
  } else if (is.list(hazard)) {
    if (!is.null(hazard$lambda)) {
      return(hazard$lambda)
    } else if (hazard$type == "negative_binomial" && !is.null(hazard$r) && !is.null(hazard$p)) {
      return(hazard$p / (hazard$r * (1 - hazard$p)))
    } else {
      cli::cli_abort("hazard must be a numeric value or a valid hazard_prior object")
    }
  } else {
    cli::cli_abort("hazard must be a numeric value or a hazard_prior object")
  }
}

#' @noRd
detect_bocpd <- function(data, type = "both", prior = NULL, hazard = NULL,
                         threshold = 0.3, truncate_run_length = NULL, ...) {
  if (is.matrix(data)) {
    return(detect_bocpd_multivariate(data, type, prior, hazard, threshold, ...))
  }
  
  n <- length(data)
  
  if (is.null(prior)) {
    prior <- normal_gamma(mu0 = 0, kappa0 = 0.01, alpha0 = 1, beta0 = 1)
  }
  
  if (is.null(hazard)) {
    hazard <- geometric_hazard(lambda = 1 / min(50, n / 10))
  }
  
  if (is.null(truncate_run_length)) {
    truncate_run_length <- min(n, 500)
  }
  
  
  H <- get_hazard_rate(hazard)
  
  R <- matrix(0, nrow = n + 1, ncol = truncate_run_length + 1)
  R[1, 1] <- 1
  
  prior_type <- if (!is.null(prior$type)) prior$type else "normal_gamma"
  
  suff_stats <- vector("list", truncate_run_length + 1)
  suff_stats[[1]] <- init_suff_stats(prior, prior_type)
  
  pred_probs <- matrix(0, nrow = n, ncol = truncate_run_length + 1)
  
  for (t in seq_len(n)) {
    x <- data[t]
    
    max_r <- min(t, truncate_run_length)
    
    for (r in 0:max_r) {
      if (R[t, r + 1] > 1e-10) {
        ss <- suff_stats[[r + 1]]
        pred_probs[t, r + 1] <- compute_predictive(x, ss, prior_type, prior)
      }
    }
    
    growth_prob <- R[t, 1:(max_r + 1)] * pred_probs[t, 1:(max_r + 1)] * (1 - H)
    
    cp_prob <- sum(R[t, 1:(max_r + 1)] * pred_probs[t, 1:(max_r + 1)] * H)
    
    evidence <- cp_prob + sum(growth_prob)
    
    if (evidence > 0) {
      R[t + 1, 1] <- cp_prob / evidence
      
      end_col <- min(max_r + 2, truncate_run_length + 1)
      n_growth <- end_col - 1
      if (n_growth > 0 && n_growth <= length(growth_prob)) {
        R[t + 1, 2:end_col] <- growth_prob[1:n_growth] / evidence
      }
      
      R[t + 1, ] <- R[t + 1, ] / sum(R[t + 1, ])
    } else {
      R[t + 1, 1] <- 1
    }
    
    new_suff_stats <- vector("list", truncate_run_length + 1)
    
    new_suff_stats[[1]] <- init_suff_stats(prior, prior_type)
    
    for (r in 1:min(max_r + 1, truncate_run_length)) {
      old_ss <- suff_stats[[r]]
      if (!is.null(old_ss)) {
        new_suff_stats[[r + 1]] <- update_suff_stats(old_ss, x, prior_type, prior)
      }
    }
    
    suff_stats <- new_suff_stats
  }
  
  prob_change <- R[2:(n + 1), 1]
  
  map_run_length <- apply(R[2:(n + 1), , drop = FALSE], 1, which.max) - 1
  
  changepoints <- which(prob_change > threshold)
  
  ci_list <- compute_bocpd_ci(R, prob_change, changepoints)
  
  list(
    changepoints = changepoints,
    n_changepoints = length(changepoints),
    posterior = R[2:(n + 1), ],
    prob_change = prob_change,
    map_run_length = map_run_length,
    confidence_intervals = ci_list,
    existence_probability = prob_change[changepoints],
    threshold = threshold,
    prior = prior,
    hazard = hazard,
    information_criterion = NULL
  )
}

#' @noRd
init_suff_stats <- function(prior, prior_type) {
  switch(prior_type,
         normal_gamma = list(
           mu = prior$mu0,
           kappa = prior$kappa0,
           alpha = prior$alpha0,
           beta = prior$beta0
         ),
         normal_known_var = list(
           mu = prior$mu0,
           sigma2 = prior$sigma0^2,
           n = 0,
           sum_x = 0
         ),
         poisson_gamma = list(
           alpha = prior$alpha0,
           beta = prior$beta0
         ),
         inverse_gamma_var = list(
           alpha = prior$alpha0,
           beta = prior$beta0,
           known_mean = prior$known_mean,
           n = 0
         ),
         list(
           mu = if (!is.null(prior$mu0)) prior$mu0 else 0,
           kappa = if (!is.null(prior$kappa0)) prior$kappa0 else 1,
           alpha = if (!is.null(prior$alpha0)) prior$alpha0 else 1,
           beta = if (!is.null(prior$beta0)) prior$beta0 else 1
         )
  )
}

#' @noRd
compute_predictive <- function(x, ss, prior_type, prior) {
  result <- switch(prior_type,
                   normal_gamma = student_t_pred(x, ss$mu, ss$kappa, ss$alpha, ss$beta),
                   normal_known_var = normal_known_var_pred(x, ss, prior),
                   poisson_gamma = poisson_gamma_pred(x, ss),
                   inverse_gamma_var = inverse_gamma_pred(x, ss),
                   student_t_pred(x, ss$mu, ss$kappa, ss$alpha, ss$beta)
  )
  
  if (is.na(result) || !is.finite(result) || result <= 0) {
    result <- 1e-300
  }
  
  result
}

#' @noRd
update_suff_stats <- function(ss, x, prior_type, prior) {
  switch(prior_type,
         normal_gamma = update_sufficient_stats_ng(ss, x),
         normal_known_var = update_sufficient_stats_nkv(ss, x, prior),
         poisson_gamma = update_sufficient_stats_pg(ss, x),
         inverse_gamma_var = update_sufficient_stats_ig(ss, x),
         update_sufficient_stats_ng(ss, x)
  )
}

#' @noRd
student_t_pred <- function(x, mu, kappa, alpha, beta) {
  if (is.null(kappa) || is.null(alpha) || is.null(beta)) {
    return(1e-300)
  }
  
  if (!is.finite(kappa) || !is.finite(alpha) || !is.finite(beta)) {
    return(1e-300)
  }
  
  if (kappa <= 0 || alpha <= 0 || beta <= 0) {
    return(1e-300)
  }
  
  df <- 2 * alpha
  scale <- sqrt(beta * (kappa + 1) / (alpha * kappa))
  
  if (scale <= 0 || !is.finite(scale)) {
    scale <- 1
  }
  
  dt((x - mu) / scale, df = df) / scale
}

#' @noRd
normal_known_var_pred <- function(x, ss, prior) {
  sigma2_obs <- prior$known_var
  sigma0_2 <- prior$sigma0^2
  
  if (ss$n == 0) {
    pred_var <- sigma0_2 + sigma2_obs
    pred_mean <- prior$mu0
  } else {
    precision_prior <- 1 / sigma0_2
    precision_data <- ss$n / sigma2_obs
    precision_post <- precision_prior + precision_data
    
    post_var <- 1 / precision_post
    post_mean <- (prior$mu0 * precision_prior + ss$sum_x / sigma2_obs) / precision_post
    
    pred_var <- post_var + sigma2_obs
    pred_mean <- post_mean
  }
  
  dnorm(x, mean = pred_mean, sd = sqrt(pred_var))
}

#' @noRd
poisson_gamma_pred <- function(x, ss) {
  if (x < 0 || x != round(x)) {
    return(1e-300)
  }
  
  r <- ss$alpha
  p <- ss$beta / (ss$beta + 1)
  
  dnbinom(x, size = r, prob = p)
}

#' @noRd
inverse_gamma_pred <- function(x, ss) {
  df <- 2 * ss$alpha
  if (df <= 0) df <- 1
  
  scale <- sqrt(ss$beta / ss$alpha)
  if (scale <= 0 || !is.finite(scale)) scale <- 1
  
  dt((x - ss$known_mean) / scale, df = df) / scale
}

#' @noRd
update_sufficient_stats_ng <- function(ss, x) {
  kappa_new <- ss$kappa + 1
  mu_new <- (ss$kappa * ss$mu + x) / kappa_new
  alpha_new <- ss$alpha + 0.5
  beta_new <- ss$beta + ss$kappa * (x - ss$mu)^2 / (2 * kappa_new)
  
  list(
    mu = mu_new,
    kappa = kappa_new,
    alpha = alpha_new,
    beta = beta_new
  )
}

#' @noRd
update_sufficient_stats_nkv <- function(ss, x, prior) {
  list(
    mu = ss$mu,
    sigma2 = ss$sigma2,
    n = ss$n + 1,
    sum_x = ss$sum_x + x
  )
}

#' @noRd
update_sufficient_stats_pg <- function(ss, x) {
  list(
    alpha = ss$alpha + x,
    beta = ss$beta + 1
  )
}

#' @noRd
update_sufficient_stats_ig <- function(ss, x) {
  list(
    alpha = ss$alpha + 0.5,
    beta = ss$beta + 0.5 * (x - ss$known_mean)^2,
    known_mean = ss$known_mean,
    n = ss$n + 1
  )
}

#' @noRd
compute_bocpd_ci <- function(R, prob_change, changepoints) {
  if (length(changepoints) == 0) return(list())
  
  n <- length(prob_change)
  ci_list <- vector("list", length(changepoints))
  
  for (i in seq_along(changepoints)) {
    cp <- changepoints[i]
    
    window <- max(1, cp - 20):min(n, cp + 20)
    probs <- prob_change[window]
    
    if (sum(probs) > 0) {
      probs <- probs / sum(probs)
      cumprobs <- cumsum(probs)
      
      lower_idx <- window[which(cumprobs >= 0.025)[1]]
      upper_idx <- window[which(cumprobs >= 0.975)[1]]
      
      if (is.na(lower_idx)) lower_idx <- window[1]
      if (is.na(upper_idx)) upper_idx <- window[length(window)]
      
      ci_list[[i]] <- list(
        estimate = cp,
        lower = lower_idx,
        upper = upper_idx,
        se = sd(window * probs)
      )
    } else {
      ci_list[[i]] <- list(
        estimate = cp,
        lower = cp,
        upper = cp,
        se = NA
      )
    }
  }
  
  ci_list
}

#' @noRd
detect_bocpd_multivariate <- function(data, type, prior, hazard, threshold, ...) {
  n <- nrow(data)
  d <- ncol(data)
  
  if (is.null(prior)) {
    prior <- normal_wishart(mu0 = rep(0, d))
  }
  
  if (is.null(hazard)) {
    hazard <- geometric_hazard(lambda = 1 / min(50, n / 10))
  }
  
  truncate_run_length <- min(n, 200)
  
  H <- get_hazard_rate(hazard)
  
  R <- matrix(0, nrow = n + 1, ncol = truncate_run_length + 1)
  R[1, 1] <- 1
  
  suff_stats <- vector("list", truncate_run_length + 1)
  suff_stats[[1]] <- list(
    mu = prior$mu0,
    kappa = prior$kappa0,
    nu = prior$nu0,
    Psi = prior$Psi0
  )
  
  for (t in seq_len(n)) {
    x <- data[t, ]
    max_r <- min(t, truncate_run_length)
    
    pred_probs <- numeric(max_r + 1)
    
    for (r in 0:max_r) {
      if (R[t, r + 1] > 1e-10) {
        ss <- suff_stats[[r + 1]]
        pred_probs[r + 1] <- mvt_pred(x, ss$mu, ss$kappa, ss$nu, ss$Psi)
      }
    }
    
    growth_prob <- R[t, 1:(max_r + 1)] * pred_probs * (1 - H)
    cp_prob <- sum(R[t, 1:(max_r + 1)] * pred_probs * H)
    
    evidence <- cp_prob + sum(growth_prob)
    
    if (evidence > 0) {
      R[t + 1, 1] <- cp_prob / evidence
      
      end_col <- min(max_r + 2, truncate_run_length + 1)
      n_growth <- end_col - 1
      if (n_growth > 0 && n_growth <= length(growth_prob)) {
        R[t + 1, 2:end_col] <- growth_prob[1:n_growth] / evidence
      }
      
      R[t + 1, ] <- R[t + 1, ] / sum(R[t + 1, ])
    } else {
      R[t + 1, 1] <- 1
    }
    
    new_suff_stats <- vector("list", truncate_run_length + 1)
    new_suff_stats[[1]] <- list(
      mu = prior$mu0,
      kappa = prior$kappa0,
      nu = prior$nu0,
      Psi = prior$Psi0
    )
    
    for (r in 1:min(max_r + 1, truncate_run_length)) {
      old_ss <- suff_stats[[r]]
      if (!is.null(old_ss)) {
        new_suff_stats[[r + 1]] <- update_sufficient_stats_nw(old_ss, x)
      }
    }
    
    suff_stats <- new_suff_stats
  }
  
  prob_change <- R[2:(n + 1), 1]
  changepoints <- which(prob_change > threshold)
  ci_list <- compute_bocpd_ci(R, prob_change, changepoints)
  
  list(
    changepoints = changepoints,
    n_changepoints = length(changepoints),
    posterior = R[2:(n + 1), ],
    prob_change = prob_change,
    map_run_length = apply(R[2:(n + 1), , drop = FALSE], 1, which.max) - 1,
    confidence_intervals = ci_list,
    existence_probability = if (length(changepoints) > 0) prob_change[changepoints] else numeric(0),
    threshold = threshold,
    information_criterion = NULL
  )
}

#' @noRd
mvt_pred <- function(x, mu, kappa, nu, Psi) {
  d <- length(x)
  
  df <- nu - d + 1
  if (df <= 0) df <- 1
  
  scale_matrix <- (kappa + 1) / (kappa * df) * Psi
  
  diff <- x - mu
  
  tryCatch({
    scale_inv <- solve(scale_matrix)
    mahal <- sum(diff * (scale_inv %*% diff))
    
    log_det <- determinant(scale_matrix, logarithm = TRUE)$modulus
    
    log_c <- lgamma((df + d) / 2) - lgamma(df / 2) -
      (d / 2) * log(df * pi) - 0.5 * log_det
    log_dens <- log_c - ((df + d) / 2) * log(1 + mahal / df)
    
    exp(log_dens)
  }, error = function(e) {
    1e-300
  })
}

#' @noRd
update_sufficient_stats_nw <- function(ss, x) {
  d <- length(x)
  
  kappa_new <- ss$kappa + 1
  mu_new <- (ss$kappa * ss$mu + x) / kappa_new
  nu_new <- ss$nu + 1
  
  diff <- x - ss$mu
  Psi_new <- ss$Psi + ss$kappa / kappa_new * outer(diff, diff)
  
  list(
    mu = mu_new,
    kappa = kappa_new,
    nu = nu_new,
    Psi = Psi_new
  )
}

#' Shiryaev-Roberts Changepoint Detection
#'
#' @description
#' Implements the Shiryaev-Roberts procedure, which is asymptotically optimal
#' for detecting changes with minimal detection delay.
#'
#' @param data Numeric vector
#' @param type Type of change to detect
#' @param prior Prior specification
#' @param hazard Hazard prior
#' @param threshold Detection threshold
#' @param mu0 Pre-change mean (if known)
#' @param mu1 Post-change mean (if known)
#' @param sigma Known standard deviation (if applicable)
#' @param ... Additional arguments
#'
#' @return List with changepoints and statistics
#'
#' @references
#' Shiryaev, A. N. (1963). On Optimum Methods in Quickest Detection Problems.
#' Theory of Probability and Its Applications.
#'
#' @examples
#' data <- c(rnorm(100), rnorm(100, mean = 1))
#' result <- shiryaev_roberts(data)
#'
#' @export
shiryaev_roberts <- function(data, type = "mean", prior = NULL, hazard = NULL,
                             threshold = 100, mu0 = NULL, mu1 = NULL,
                             sigma = NULL, ...) {
  detect_shiryaev(data, type, prior, hazard, threshold, mu0, mu1, sigma, ...)
}

#' @noRd
detect_shiryaev <- function(data, type = "mean", prior = NULL, hazard = NULL,
                            threshold = 100, mu0 = NULL, mu1 = NULL,
                            sigma = NULL, ...) {
  n <- length(data)
  
  if (is.null(mu0)) {
    ref_n <- min(30, n / 4)
    mu0 <- mean(data[1:ref_n])
  }
  
  if (is.null(sigma)) {
    ref_n <- min(30, n / 4)
    sigma <- sd(data[1:ref_n])
    if (sigma <= 0) sigma <- 1
  }
  
  if (is.null(mu1)) {
    mu1 <- mu0 + sigma
  }
  
  R <- numeric(n)
  log_R <- numeric(n)
  
  alarm <- FALSE
  alarm_time <- NA
  
  for (t in seq_len(n)) {
    x <- data[t]
    
    log_lr <- dnorm(x, mean = mu1, sd = sigma, log = TRUE) -
      dnorm(x, mean = mu0, sd = sigma, log = TRUE)
    
    if (t == 1) {
      log_R[t] <- log_lr
    } else {
      log_R[t] <- log(1 + exp(log_R[t-1])) + log_lr
    }
    
    R[t] <- exp(log_R[t])
    
    if (R[t] > threshold && !alarm) {
      alarm <- TRUE
      alarm_time <- t
    }
  }
  
  prob_change <- R / (1 + R)
  
  changepoints <- integer(0)
  if (alarm) {
    peaks <- find_peaks(R, threshold)
    changepoints <- peaks
  }
  
  if (!is.null(hazard)) {
    H <- get_hazard_rate(hazard)
    pi_t <- cumsum(rep(H, n))
    pi_t <- pmin(pi_t, 0.99)
    
    post_odds <- R * pi_t / (1 - pi_t)
    prob_change <- post_odds / (1 + post_odds)
    prob_change <- pmin(prob_change, 1)
  }
  
  list(
    changepoints = changepoints,
    n_changepoints = length(changepoints),
    R = R,
    log_R = log_R,
    prob_change = prob_change,
    alarm = alarm,
    alarm_time = alarm_time,
    threshold = threshold,
    mu0 = mu0,
    mu1 = mu1,
    sigma = sigma,
    information_criterion = NULL
  )
}

#' @noRd
find_peaks <- function(x, threshold = 0) {
  n <- length(x)
  peaks <- integer(0)
  
  for (i in 2:(n-1)) {
    if (x[i] > x[i-1] && x[i] > x[i+1] && x[i] > threshold) {
      peaks <- c(peaks, i)
    }
  }
  
  if (n >= 1 && x[1] > threshold && (n < 2 || x[1] > x[2])) {
    peaks <- c(1, peaks)
  }
  if (n >= 2 && x[n] > threshold && x[n] > x[n-1]) {
    peaks <- c(peaks, n)
  }
  
  unique(sort(peaks))
}

#' Create Online Regime Detector
#'
#' @description
#' Creates a detector object for online (sequential) changepoint detection.
#' The detector maintains state and can be updated incrementally as new
#' observations arrive.
#'
#' @param method Detection method: "bocpd", "cusum", or "shiryaev"
#' @param prior Prior specification for Bayesian methods
#' @param hazard Hazard prior for changepoint occurrence
#' @param threshold Detection threshold (probability or statistic value)
#' @param ... Additional method-specific parameters
#'
#' @return An object of class "regime_detector"
#'
#' @examples
#' detector <- regime_detector(method = "bocpd",
#'                             prior = normal_gamma(),
#'                             threshold = 0.5)
#'
#' \donttest{
#' for (x in rnorm(100)) {
#'   detector <- update(detector, x)
#'   if (detector$last_result$alarm) {
#'     message("Change detected at observation ", detector$last_result$t)
#'     detector <- reset(detector)
#'   }
#' }
#' }
#'
#' @export
regime_detector <- function(method = c("bocpd", "cusum", "shiryaev"),
                            prior = NULL, hazard = NULL,
                            threshold = NULL, ...) {
  method <- match.arg(method)
  
  if (is.null(prior)) {
    prior <- normal_gamma(mu0 = 0, kappa0 = 0.01, alpha0 = 1, beta0 = 1)
  }
  
  if (is.null(hazard)) {
    hazard <- geometric_hazard(lambda = 0.01)
  }
  
  if (is.null(threshold)) {
    threshold <- switch(method,
                        bocpd = 0.5,
                        cusum = 5,
                        shiryaev = 100
    )
  }
  
  prior_type <- if (!is.null(prior$type)) prior$type else "normal_gamma"
  
  state <- switch(method,
                  bocpd = list(
                    R = 1,
                    suff_stats = init_suff_stats(prior, prior_type),
                    t = 0,
                    max_run_length = 200,
                    prior_type = prior_type
                  ),
                  cusum = list(
                    S_plus = 0,
                    S_minus = 0,
                    t = 0,
                    mu0 = NULL,
                    sigma = NULL
                  ),
                  shiryaev = list(
                    R = 0,
                    log_R = -Inf,
                    t = 0,
                    mu0 = NULL,
                    mu1 = NULL,
                    sigma = NULL
                  )
  )
  
  structure(
    list(
      method = method,
      prior = prior,
      hazard = hazard,
      threshold = threshold,
      state = state,
      history = list(
        prob_change = numeric(0),
        statistic = numeric(0),
        alarms = integer(0)
      ),
      params = list(...),
      last_result = NULL
    ),
    class = c("regime_detector", "list")
  )
}

#' Update Online Detector with New Observation
#'
#' @param object A regime_detector object
#' @param x New observation (scalar or vector for multivariate)
#' @param ... Additional arguments
#'
#' @return The updated regime_detector object with results in \code{$last_result}
#'
#' @method update regime_detector
#' @export
update.regime_detector <- function(object, x, ...) {
  detector <- object
  method <- detector$method
  
  result <- switch(method,
                   bocpd = update_bocpd_online(detector, x),
                   cusum = update_cusum_online(detector, x),
                   shiryaev = update_shiryaev_online(detector, x)
  )
  
  updated_detector <- result$detector
  updated_detector$last_result <- result[names(result) != "detector"]
  
  updated_detector
}

#' @noRd
update_bocpd_online <- function(detector, x) {
  state <- detector$state
  prior <- detector$prior
  hazard <- detector$hazard
  threshold <- detector$threshold
  prior_type <- state$prior_type
  
  state$t <- state$t + 1
  t <- state$t
  
  H <- get_hazard_rate(hazard)
  max_r <- min(t, state$max_run_length)
  
  R <- state$R
  if (length(R) < max_r + 1) {
    R <- c(R, rep(0, max_r + 1 - length(R)))
  }
  
  pred_prob <- compute_predictive(x, state$suff_stats, prior_type, prior)
  
  pred_probs <- rep(pred_prob, max_r + 1)
  
  growth_prob <- R[1:(max_r + 1)] * pred_probs[1:(max_r + 1)] * (1 - H)
  cp_prob <- sum(R[1:(max_r + 1)] * pred_probs[1:(max_r + 1)] * H)
  
  evidence <- cp_prob + sum(growth_prob)
  
  new_R <- numeric(max_r + 2)
  if (evidence > 0) {
    new_R[1] <- cp_prob / evidence
    new_R[2:(max_r + 2)] <- growth_prob / evidence
  } else {
    new_R[1] <- 1
  }
  
  state$suff_stats <- update_suff_stats(state$suff_stats, x, prior_type, prior)
  state$R <- new_R
  
  prob_change <- new_R[1]
  alarm <- prob_change > threshold
  
  detector$history$prob_change <- c(detector$history$prob_change, prob_change)
  if (alarm) {
    detector$history$alarms <- c(detector$history$alarms, t)
  }
  
  detector$state <- state
  
  list(
    detector = detector,
    t = t,
    prob_change = prob_change,
    alarm = alarm,
    run_length_dist = new_R
  )
}

#' @noRd
update_cusum_online <- function(detector, x) {
  state <- detector$state
  threshold <- detector$threshold
  
  state$t <- state$t + 1
  t <- state$t
  
  if (t <= 10) {
    if (is.null(state$mu0)) {
      state$mu0 <- x
      state$sigma <- 1
    } else {
      old_mean <- state$mu0
      state$mu0 <- old_mean + (x - old_mean) / t
      if (t > 1) {
        state$sigma <- sqrt(((t - 2) * state$sigma^2 + (x - old_mean) * (x - state$mu0)) / (t - 1))
      }
    }
  }
  
  z <- (x - state$mu0) / max(state$sigma, 0.001)
  
  state$S_plus <- max(0, state$S_plus + z)
  state$S_minus <- max(0, state$S_minus - z)
  
  statistic <- max(state$S_plus, state$S_minus)
  alarm <- statistic > threshold
  
  detector$history$statistic <- c(detector$history$statistic, statistic)
  if (alarm) {
    detector$history$alarms <- c(detector$history$alarms, t)
  }
  
  detector$state <- state
  
  list(
    detector = detector,
    t = t,
    statistic = statistic,
    S_plus = state$S_plus,
    S_minus = state$S_minus,
    alarm = alarm
  )
}

#' @noRd
update_shiryaev_online <- function(detector, x) {
  state <- detector$state
  threshold <- detector$threshold
  
  state$t <- state$t + 1
  t <- state$t
  
  if (t <= 10) {
    if (is.null(state$mu0)) {
      state$mu0 <- x
      state$sigma <- 1
      state$mu1 <- x + 1
    } else {
      old_mean <- state$mu0
      state$mu0 <- old_mean + (x - old_mean) / t
      if (t > 1) {
        state$sigma <- sqrt(((t - 2) * state$sigma^2 + (x - old_mean) * (x - state$mu0)) / (t - 1))
      }
      state$mu1 <- state$mu0 + max(state$sigma, 0.1)
    }
  }
  
  log_lr <- dnorm(x, mean = state$mu1, sd = state$sigma, log = TRUE) -
    dnorm(x, mean = state$mu0, sd = state$sigma, log = TRUE)
  
  if (is.infinite(state$log_R)) {
    state$log_R <- log_lr
  } else {
    state$log_R <- log(1 + exp(state$log_R)) + log_lr
  }
  
  state$R <- exp(state$log_R)
  alarm <- state$R > threshold
  
  detector$history$statistic <- c(detector$history$statistic, state$R)
  if (alarm) {
    detector$history$alarms <- c(detector$history$alarms, t)
  }
  
  detector$state <- state
  
  list(
    detector = detector,
    t = t,
    R = state$R,
    alarm = alarm
  )
}

#' Reset a detector to its initial state
#'
#' @param object A detector object (e.g., from BOCPD or Shiryaev-Roberts)
#' @param ... Additional arguments
#'
#' @return The reset detector object
#'
#' @export
reset <- function(object, ...) {
  UseMethod("reset")
}

#' @describeIn reset Reset method for regime_detector
#' @export
reset.regime_detector <- function(object, ...) {
  detector <- object
  method <- detector$method
  prior <- detector$prior
  prior_type <- if (!is.null(prior$type)) prior$type else "normal_gamma"
  
  detector$state <- switch(method,
                           bocpd = list(
                             R = 1,
                             suff_stats = init_suff_stats(prior, prior_type),
                             t = detector$state$t,
                             max_run_length = 200,
                             prior_type = prior_type
                           ),
                           cusum = list(
                             S_plus = 0,
                             S_minus = 0,
                             t = detector$state$t,
                             mu0 = NULL,
                             sigma = NULL
                           ),
                           shiryaev = list(
                             R = 0,
                             log_R = -Inf,
                             t = detector$state$t,
                             mu0 = NULL,
                             mu1 = NULL,
                             sigma = NULL
                           )
  )
  
  detector$last_result <- NULL
  
  detector
}

#' @export
print.regime_detector <- function(x, ...) {
  cat("Online Regime Detector\n")
  cat("======================\n")
  cat("Method:", x$method, "\n")
  cat("Threshold:", x$threshold, "\n")
  cat("Observations processed:", x$state$t, "\n")
  cat("Alarms triggered:", length(x$history$alarms), "\n")
  
  if (length(x$history$alarms) > 0) {
    cat("Alarm times:", paste(head(x$history$alarms, 10), collapse = ", "))
    if (length(x$history$alarms) > 10) cat(", ...")
    cat("\n")
  }
  
  invisible(x)
}