#' Search for exact full-length matches
#'
#' @description \code{vs_search_exact} searches for exact full-length matches of
#' query sequences in a database of target sequences using \code{VSEARCH}.
#'
#' @param fastx_input (Required). A FASTA/FASTQ file path or FASTA/FASTQ tibble
#' object containing the query sequences. See \emph{Details}.
#' @param database (Required). A FASTA/FASTQ file path or FASTA/FASTQ tibble
#' object containing the target sequences.
#' @param userout (Optional). A character string specifying the name of the
#' output file for the alignment results. If \code{NULL} (default), no output is
#' written to a file and the results are returned as a tibble with the columns
#' specified in \code{userfields}. See \emph{Details}.
#' @param otutabout (Optional). A character string specifying the name of the
#' output file in an OTU table format. If \code{NULL} (default), no output is
#' written to a file. If \code{TRUE}, the output is returned as a tibble. See
#' \emph{Details}.
#' @param userfields (Optional). Fields to include in the output file. Defaults
#' to \code{"query+target+id+alnlen+mism+opens+qlo+qhi+tlo+thi+evalue+bits"}.
#' See \emph{Details}.
#' @param strand (Optional). Specifies which strand to consider when comparing
#' sequences. Can be either \code{"plus"} (default) or \code{"both"}.
#' @param threads (Optional). Number of computational threads to be used by
#' \code{VSEARCH}. Defaults to \code{1}.
#' @param vsearch_options (Optional). A character string of additional arguments
#' to pass to \code{VSEARCH}. Defaults to \code{NULL}. See \emph{Details}.
#' @param tmpdir (Optional). Path to the directory where temporary files should
#' be written when tables are used as input or output. Defaults to
#' \code{NULL}, which resolves to the session-specific temporary directory
#' (\code{tempdir()}).
#'
#' @details
#' Identifies exact full-length matches between query and target sequences
#' using \code{VSEARCH}. Only 100% identical matches are reported, ensuring high
#' specificity and making this command much faster than
#' \code{\link{vs_usearch_global}}.
#'
#' \code{fastx_input} and \code{database} can either be file paths to a
#' FASTA/FASTQ files or FASTA/FASTQ objects. FASTA objects are tibbles that
#' contain the columns \code{Header} and \code{Sequence}, see
#' \code{\link[microseq]{readFasta}}. FASTQ objects are tibbles that contain the
#' columns \code{Header}, \code{Sequence}, and \code{Quality}, see
#' \code{\link[microseq]{readFastq}}.
#'
#' \code{userfields} specifies the fields to include in the output file. Fields
#' must be given as a character string separated by \code{"+"}. The default
#' value of \code{userfields} equals
#' \code{"query+target+id+alnlen+mism+opens+qlo+qhi+tlo+thi+evalue+bits"}, which
#' gives a blast-like tab-separated format of twelve fields. See the
#' 'Userfields' section in the \code{VSEARCH} manual for more information.
#'
#' \code{otutabout} gives the option to output the results in an OTU
#' table format with tab-separated columns. When writing to a file, the first
#' line starts with the string "#OTU ID", followed by a tab-separated list of
#' all sample identifiers (formatted as "sample=X"). Each subsequent line,
#' corresponding to an OTU, begins with the OTU identifier and is followed by
#' tab-separated abundances for that OTU in each sample. If \code{otutabout} is
#' a character string, the output is written to the specified file. If
#' \code{otutabout} is \code{TRUE}, the function returns the OTU table as a
#' tibble, where the first column is named \code{otu_id} instead of "#OTU ID".
#'
#' \code{vsearch_options} allows users to pass additional command-line arguments
#' to \code{VSEARCH} that are not directly supported by this function. Refer to
#' the \code{VSEARCH} manual for more details.
#'
#' @return A tibble or \code{NULL}.
#'
#' If \code{userout} is \code{NULL} a tibble containing the alignment results
#' with the fields specified by \code{userfields} is returned. If \code{userout}
#' is specified the alignment results are written to the
#' specified file, and no tibble is returned.
#'
#' If \code{otutabout} is \code{TRUE}, an OTU table is returned as a tibble.
#' If \code{otutabout} is a character string, the output is written to the file,
#' and no tibble is returned.
#'
#' If neither \code{userout} nor \code{otutabout} is specified, a tibble
#' containing the alignment results is returned.
#'
#' @seealso \code{\link{vs_usearch_global}}
#'
#' @examples
#' \dontrun{
#' # You would typically use something else as database
#' query_file <- file.path(file.path(path.package("Rsearch"), "extdata"),
#'                         "small.fasta")
#' db <- query_file
#'
#' # Search for exact full-length matches with default parameters, with file as output
#' vs_search_exact(fastx_input = query_file,
#'                 database = db,
#'                 userout = "delete_me.txt")
#'
#' # Read results, and give column names
#' result.tbl <- read.table("delete_me.txt",
#'                          sep = "\t",
#'                          header = FALSE,
#'                          col.names = c("query", "target", "id", "alnlen",
#'                                        "mism", "opens", "qlo", "qhi",
#'                                        "tlo", "thi", "evalue", "bits"))
#' }
#'
#' @references \url{https://github.com/torognes/vsearch}
#'
#' @aliases vs_search_exact search_exact
#'
#' @export
#'
vs_search_exact <- function(fastx_input,
                            database,
                            userout = NULL,
                            otutabout = NULL,
                            userfields = "query+target+id+alnlen+mism+opens+qlo+qhi+tlo+thi+evalue+bits",
                            strand = "plus",
                            threads = 1,
                            vsearch_options = NULL,
                            tmpdir = NULL) {

  # Check if vsearch is available
  vsearch_executable <- options("Rsearch.vsearch_executable")[[1]]
  vsearch_available(vsearch_executable)

  # Set temporary directory if not provided
  if (is.null(tmpdir)) tmpdir <- tempdir()

  # Validate strand
  if (!strand %in% c("plus", "both")) {
    stop("Invalid value for 'strand'. Choose from 'plus' or 'both'.")
  }

  # Ensure only one output format is specified
  if (!is.null(userout) && !is.null(otutabout)) {
    stop("Only one of 'userout' or 'otutabout' can be specified.")
  }

  # Create empty vector for collecting temporary files
  temp_files <- character()

  # Set up cleanup of temporary files
  on.exit({
    if (length(temp_files) > 0 && is.character(temp_files)) {
      existing_files <- temp_files[file.exists(temp_files)]
      if (length(existing_files) > 0) {
        file.remove(existing_files)
      }
    }
  }, add = TRUE)

  # Handle input query sequences
  if (!is.character(fastx_input)){
    if ("Quality" %in% colnames(fastx_input)){

      # Validate tibble
      required_cols <- c("Header", "Sequence", "Quality")
      if (!all(required_cols %in% colnames(fastx_input))) {
        stop("FASTQ object must contain columns: Header, Sequence, Quality")
      }

      temp_file_fastx <- tempfile(pattern = "fastx_input",
                                  tmpdir = tmpdir,
                                  fileext = ".fq")
      temp_files <- c(temp_files, temp_file_fastx)
      microseq::writeFastq(fastx_input, temp_file_fastx)

      fastx_file <- temp_file_fastx

    } else {

      # Validate tibble
      required_cols <- c("Header", "Sequence")
      if (!all(required_cols %in% colnames(fastx_input))) {
        stop("FASTA object must contain columns: Header and Sequence")
      }

      temp_file_fastx <- tempfile(pattern = "fastx_input",
                                  tmpdir = tmpdir,
                                  fileext = ".fa")
      temp_files <- c(temp_files, temp_file_fastx)
      microseq::writeFasta(fastx_input, temp_file_fastx)

      fastx_file <- temp_file_fastx

    }
  } else {
    if (!file.exists(fastx_input)) stop("Cannot find input file: ", fastx_input)

    fastx_file <- fastx_input
  }

  # Handle input target sequences
  if (!is.character(database)){
    if ("Quality" %in% colnames(database)){

      # Validate tibble
      required_cols <- c("Header", "Sequence", "Quality")
      if (!all(required_cols %in% colnames(database))) {
        stop("FASTQ object must contain columns: Header, Sequence, Quality")
      }

      temp_file_db <- tempfile(pattern = "db_input",
                               tmpdir = tmpdir,
                               fileext = ".fq")
      temp_files <- c(temp_files, temp_file_db)
      microseq::writeFastq(database, temp_file_db)

      db_file <- temp_file_db

    } else {

      # Validate tibble
      required_cols <- c("Header", "Sequence")
      if (!all(required_cols %in% colnames(database))) {
        stop("FASTA object must contain columns: Header and Sequence")
      }

      temp_file_db <- tempfile(pattern = "db_input",
                               tmpdir = tmpdir,
                               fileext = ".fa")
      temp_files <- c(temp_files, temp_file_db)
      microseq::writeFasta(database, temp_file_db)

      db_file <- temp_file_db

    }
  } else {
    if (!file.exists(database)) stop("Cannot find input file: ", database)

    db_file <- database
  }

  # Determine output file based on user input
  if (!is.null(userout)) {
    outfile <- userout
  } else if (!is.null(otutabout)) {
    outfile <- ifelse(is.character(otutabout), otutabout, tempfile(pattern = "otutable",
                                                                   tmpdir = tmpdir,
                                                                   fileext = ".tsv"))
  } else {
    outfile <- tempfile(pattern = "userout",
                        tmpdir = tmpdir,
                        fileext = ".txt")
  }

  if (is.null(userout) && (is.null(otutabout) || !is.character(otutabout))) {
    temp_files <- c(temp_files, outfile)
  }

  # Normalize file paths
  fastx_file <- normalizePath(fastx_file)
  db_file <- normalizePath(db_file)

  # Build argument string for command line
  args <- c("--search_exact", shQuote(fastx_file),
            "--db", shQuote(db_file),
            "--threads", threads,
            "--strand", strand)

  if (!is.null(userout)) {
    args <- c(args, "--userout", outfile, "--userfields", userfields)
  } else if (!is.null(otutabout)) {
    args <- c(args, "--otutabout", outfile)
  } else {
    args <- c(args, "--userout", outfile, "--userfields", userfields) # Default output
  }

  # Add additional arguments if specified
  if (!is.null(vsearch_options)) {
    args <- c(args, vsearch_options)
  }

  # Run VSEARCH
  vsearch_output <- system2(command = vsearch_executable,
                            args = args,
                            stdout = TRUE,
                            stderr = TRUE)

  # Check for VSEARCH failure
  check_vsearch_status(vsearch_output, args)

  # Determine return output
  if (!is.null(userout)) {
    return(invisible(NULL)) # No return if userout is specified
  } else if (!is.null(otutabout)) {
    if (is.character(otutabout)) {
      return(invisible(NULL)) # File output only
    } else {
      df <- suppressMessages(readr::read_delim(outfile))
      colnames(df)[1] <- "otu_id"
      return(df) # Return as tibble
    }
  } else {
    userout_df <- suppressMessages(readr::read_delim(outfile, col_names = FALSE))
    columns <- unlist(strsplit(userfields, "\\+"))
    colnames(userout_df) <- columns
    return(userout_df)
  }
}
