% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spatial_kde.R
\name{spatial_kde}
\alias{spatial_kde}
\title{Kernel Density Estimation (Heatmap) for occurrence data}
\usage{
spatial_kde(
  occ,
  long = "decimalLongitude",
  lat = "decimalLatitude",
  radius = 0.2,
  resolution = NULL,
  buffer_extent = 500,
  crs = "epsg:4326",
  raster_ref = NULL,
  kernel = "quartic",
  scaled = TRUE,
  decay = 1,
  mask = NULL,
  zero_as_NA = FALSE,
  weights = NULL
)
}
\arguments{
\item{occ}{(data.frame, data.table, or SpatVector) a data frame or SpatVector containing the occurrences. Must contain columns longitude and latitude.}

\item{long}{(character) the name of the column in \code{occ} that contains the
longitude values. Default is \code{"decimalLongitude"}.}

\item{lat}{(character) the name of the column in \code{occ} that contains the
latitude values. Default is \code{"decimalLatitude"}.}

\item{radius}{(numeric) a positive numeric value specifying the smoothing
radius for the kernel density estimate. This parameter determines the
circular neighborhood around each point where that point will have an
influence. See details. Default is 0.2.}

\item{resolution}{(numeric) a positive numeric value specifying the resolution
(in degrees or meters, depending on the \code{crs} provided) of the resultant
raster heatmap. Required if \code{raster_ref} is not provided. Default is NULL.}

\item{buffer_extent}{(numeric) width of the buffer (in kilometers) to draw
around the occurrences to define the area for computing the heatmap. Default
is 500.}

\item{crs}{(character) the coordinate reference system of the raster heatmap
(see ?terra::crs). Default is "epsg:4326". Only applicable if \code{raster_ref}
is not provided.}

\item{raster_ref}{(SpatRaster) an optional  raster to use as reference for
resolution, CRS, and extent. Default is NULL.}

\item{kernel}{(character) type of kernel to use. Available kernerls are
"uniform", "quartic", "triweight", "epanechnikov", or "triangular". Default
is "quartic".}

\item{scaled}{(logical) whether to scale output values to vary between 0 and
\enumerate{
\item Default is TRUE.
}}

\item{decay}{(numeric) decay parameter for "triangular" kernel. Only
applicable if \code{kernel = triangular}.}

\item{mask}{(SpatRaster or SpatExtent) optional spatial object to define the
extent of the area for the heatmap. Default is NULL, in which case the
extent is derived from \code{raster_ref} (if provided) or from the convex hull of
occurrences plus \code{buffer_extent}.}

\item{zero_as_NA}{(logical) whether to convert regions with value 0 to NA.
Default is FALSE.}

\item{weights}{(numeric) optional vector of weights for individual points.
Must be the same length as the number of occurrences in \code{occ}. Default is
NULL.}
}
\value{
A SpatRaster containing the kernel density values.
}
\description{
This function creates density heatmaps using kernel density estimation.
The algorithm is inspired by the \strong{SpatialKDE} R package and the "Heatmap"
tool from QGIS.
Each occurrence contributes to the density surface within a circular
neighborhood defined by a specified radius.
}
\details{
The \code{radius} parameter controls how far the influence of each observation
extends. Smaller values produce fine-grained peaks; larger values produce
smoother, more spread-out heatmaps. Units depend on the CRS: degrees for
geographic coordinates (default), meters for projected coordinates.

If \code{raster_ref} is not provided, the extent is calculated from the convex
hull of \code{occ} plus \code{buffer_extent}.

Kernels define the weight decay of points:
"uniform" = constant, "quartic"/"triweight"/"epanechnikov" = smooth, and
"triangular" = linear decay (using \code{decay} parameter).
}
\examples{
# Load example data
data("occ_flagged", package = "RuHere")
# Remove flagged records
occ <- remove_flagged(occ_flagged)
# Generate heatmap
heatmap <- spatial_kde(occ = occ, resolution = 0.25, buffer_extent = 50,
                       radius = 2)
# Plot heatmap with terra
terra::plot(heatmap)
# Plot heatmap with ggplot
ggmap_here(occ = occ, heatmap = heatmap)
}
\references{
\itemize{
\item Hart, T., & Zandbergen, P. (2014). Kernel density estimation and hotspot
mapping: Examining the influence of interpolation method, grid cell size,
and radius on crime forecasting. \strong{Policing: An International Journal of}
\strong{Police Strategies & Management}, 37(2), 305-323.
\item Nelson, T. A., & Boots, B. (2008). Detecting spatial hot spots in landscape
ecology. \strong{Ecography}, 31(5), 556-566.
\item Chainey, S., Tompson, L., & Uhlig, S. (2008). The utility of hotspot mapping
for predicting spatial patterns of crime. \strong{Security journal}, 21(1), 4-28.
\item Caha J (2023). SpatialKDE: Kernel Density Estimation for Spatial Data.
\strong{https://jancaha.github.io/SpatialKDE/index.html}.
}
}
