# Test data simulation functions
# These tests verify simulate_spatial_data and related functions

test_that("simulate_spatial_data returns correct structure", {
    set.seed(42)
    sim <- simulate_spatial_data(
        n_spots = 100,
        n_genes = 50,
        n_svg = 10
    )

    expect_type(sim, "list")
    expect_true("counts" %in% names(sim))
    expect_true("logcounts" %in% names(sim))
    expect_true("spatial_coords" %in% names(sim))
    expect_true("gene_info" %in% names(sim))
})

test_that("simulate_spatial_data returns correct dimensions", {
    set.seed(42)
    n_spots <- 100
    n_genes <- 50

    sim <- simulate_spatial_data(
        n_spots = n_spots,
        n_genes = n_genes,
        n_svg = 10
    )

    expect_equal(nrow(sim$counts), n_genes)
    expect_equal(nrow(sim$gene_info), n_genes)
    # n_spots may differ slightly due to grid generation
    expect_true(ncol(sim$counts) >= n_spots * 0.9)
    expect_equal(ncol(sim$counts), nrow(sim$spatial_coords))
})

test_that("simulate_spatial_data creates correct number of SVGs", {
    set.seed(42)
    n_svg <- 15

    sim <- simulate_spatial_data(
        n_spots = 100,
        n_genes = 50,
        n_svg = n_svg
    )

    expect_equal(sum(sim$gene_info$is_svg), n_svg)
})

test_that("simulate_spatial_data generates valid counts", {
    set.seed(42)
    sim <- simulate_spatial_data(
        n_spots = 100,
        n_genes = 50,
        n_svg = 10
    )

    # Counts should be non-negative integers
    expect_true(all(sim$counts >= 0))
    expect_true(all(sim$counts == floor(sim$counts)))

    # Logcounts should be finite
    expect_true(all(is.finite(sim$logcounts)))
})

test_that("different grid types work correctly", {
    for (grid in c("hexagonal", "square", "random")) {
        set.seed(42)
        sim <- simulate_spatial_data(
            n_spots = 50,
            n_genes = 20,
            n_svg = 5,
            grid_type = grid
        )

        expect_s3_class(sim$gene_info, "data.frame")
        expect_true(nrow(sim$spatial_coords) > 0)
    }
})

test_that("different pattern types are assigned correctly", {
    set.seed(42)
    sim <- simulate_spatial_data(
        n_spots = 100,
        n_genes = 50,
        n_svg = 20,
        pattern_types = c("gradient", "hotspot", "periodic", "cluster")
    )

    svg_patterns <- sim$gene_info$pattern_type[sim$gene_info$is_svg]

    # Should have some genes with each pattern type
    expect_true(any(svg_patterns == "gradient"))
    expect_true(any(svg_patterns == "hotspot"))
})

test_that("reproducibility with same seed", {
    set.seed(123)
    sim1 <- simulate_spatial_data(n_spots = 50, n_genes = 20, n_svg = 5)
    
    set.seed(123)
    sim2 <- simulate_spatial_data(n_spots = 50, n_genes = 20, n_svg = 5)

    expect_equal(sim1$counts, sim2$counts)
    expect_equal(sim1$spatial_coords, sim2$spatial_coords)
})
