% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/make.functions.R
\name{make.trait.model}
\alias{make.trait.model}
\title{Model for trait evolution across the phylogeny}
\usage{
make.trait.model(
  initial_states,
  hyb.event.fxn,
  hyb.compatibility.fxn,
  time.fxn = NULL,
  spec.fxn = NULL
)
}
\arguments{
\item{initial_states}{the initial state on the phylogeny. if simulating networks with \code{twolineages=TRUE} then a vector of length two will be required.}

\item{hyb.event.fxn}{A function that describes how the trait changes after hybridization events. See Details for more information}

\item{hyb.compatibility.fxn}{A function that describes whether hybridization events can occur between taxa based on their trait values. See Details for more information}

\item{time.fxn}{A function that describes how trait values changes over time. See Details for more information}

\item{spec.fxn}{A function that describes how trait values change at speciation events. See Details for more information}
}
\value{
A model for trait evolution to be used as the \code{trait.model} argument in a `sim.bdh function``
}
\description{
Create a model that dictates how a discrete or continuous trait evolves and affects the diversification of the phylogeny. This function creates a list that dictates how the trait affects hybridizations, how the trait is changes over time, and how the trait is inherited across speciation and hybridization events.
}
\details{
\code{hyb.event.fxn} is a function that denotes the trait value of a hybrid child after a hybridization event. The function should have the argument \code{parent_states}, a vector with the trait states of the two parents to the hybrid child and \code{inheritance}. \code{parent_states} is vector with the states of the hybrid parents while \code{inheritance} is the inheritance probability of the first lineage denoted in \code{parent_states}. The function should return a single value for the trait state of the hybrid child.

\code{hyb.compatibility.fxn} describes when hybridization events can occur between two taxa based on their trait values. The function should have the arguments \code{parent_states}.  The function should return \code{TRUE} for when a hybridization event is allowed to proceed and \code{FALSE} otherwise.

\code{time.fxn} is a function that describes how trait values change over time. The function should have the arguments \code{trait_states} and \code{timestep} in that order. \code{trait_states} is a vector containing the ploidy of all taxa while \code{timestep} is the amount of time given for ploidy evolution. The function should return a vector with the updated ploidy states of all taxa.
The default value of \code{NULL} indicates that trait values will not evolve within a lineage over time. \strong{NOTE:} Values can still change at speciation or hybridization events if allowed.

\code{spec.fxn} is a function that describes how trait values change at speciation events. The function should have the argument \code{tip_state} which has the state of the lineage just before speciation. The function should return a vector with two values, one denoting the trait of each of the two new species after the event.
The default value of \code{NULL} causes the two children lineage to inherit the same trait value as the parental lineage
}
\examples{
initial_val<-2 ## The root starts off at 2N

###function for what happens at hybridization event
hyb_e_fxn <- function(parent_states,inheritance){
 ##For allopolyploidy we add the ploidy of both parents
 return(sum(parent_states))
}

##Function for determining whether hybridization occurs
hyb_c_fxn <-function(parent_states,hybrid_state){
 ##Hybridization occurs only when the ploidy is the same
 return(parent_states[1]==parent_states[2])
}


##Function for how the trait changes over time
t_fxn <- function(trait_states,timestep){
 ##We assume that autopolyploidy occur exponentially with rate lambda
 lambda<- 2 ##Rate of autopolyploidy

 ##The number of autopolyploidy events that occur on each lineage over the timestep
 nevents<-rpois(length(trait_states),timestep)

 ##each event doubles the ploidy
 new_states<- trait_states * (2^nevents)
 return(new_states)
}

##Function for how the trait changes at speciation events
s_fxn <-function(tip_state){
 ##Ploidy doesn't change at speciation events.
 ##Both daughter lineages have the same ploidy as the parent
 return(c(tip_state,tip_state))
}

trait_model<-make.trait.model(initial_states = initial_val,
                             hyb.event.fxn = hyb_e_fxn,
                             hyb.compatibility.fxn = hyb_c_fxn,
                             time.fxn = t_fxn,
                             spec.fxn = s_fxn)

}
