% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pca_dfm.R
\name{pca_bootstrap}
\alias{pca_bootstrap}
\title{Principal Component Analysis with Bootstrap Significance Testing}
\usage{
pca_bootstrap(
  X,
  n_components = NULL,
  center = TRUE,
  scale = TRUE,
  n_boot = 200,
  block_length = NULL,
  alpha = 0.05,
  use_fdr = FALSE,
  rotation = c("varimax", "none", "oblimin"),
  verbose = FALSE
)
}
\arguments{
\item{X}{Matrix or data frame where rows are observations (time points)
and columns are variables.}

\item{n_components}{Number of principal components to extract. If NULL,
determined by eigenvalue threshold or explained variance.}

\item{center}{Logical. Center variables before PCA. Default TRUE.}

\item{scale}{Logical. Scale variables to unit variance. Default TRUE.}

\item{n_boot}{Number of bootstrap replications for significance testing.
Default 200.}

\item{block_length}{Block length for block bootstrap. If NULL, defaults
to \code{ceiling(sqrt(nrow(X)))}.}

\item{alpha}{Significance level for loading tests. Default 0.05.}

\item{use_fdr}{Logical. Apply Benjamini-Hochberg FDR correction.
Default FALSE.}

\item{rotation}{Character string specifying rotation method: "none",
"varimax", or "oblimin". Default "varimax".}

\item{verbose}{Logical for progress messages.}
}
\value{
A list of class "signaly_pca" containing:
\describe{
\item{loadings}{Matrix of factor loadings (rotated if specified)}
\item{scores}{Matrix of component scores}
\item{eigenvalues}{Vector of eigenvalues}
\item{variance_explained}{Proportion of variance explained by each
component}
\item{cumulative_variance}{Cumulative proportion of variance explained}
\item{significant_loadings}{Matrix of logical values indicating
significance}
\item{p_values}{Matrix of bootstrap p-values for loadings}
\item{thresholds}{Cutoff values for significance by component}
\item{entropy}{Shannon entropy of loadings for each component}
\item{summary_by_component}{Data frame summarizing each component}
\item{assignments}{Data frame mapping variables to their dominant
component}
}
}
\description{
Performs PCA on panel data with bootstrap-based significance testing for
factor loadings. Identifies which variables load significantly on each
principal component using a null distribution constructed via block
bootstrapping.
}
\details{
The analysis proceeds in several stages:

\strong{1. Standard PCA}: Eigendecomposition of the correlation (if scaled) or
covariance matrix to extract principal components.

\strong{2. Rotation} (optional): Varimax rotation maximizes the variance of
squared loadings within components, producing cleaner simple structure.
Oblimin allows correlated factors.

\strong{3. Bootstrap Significance Testing}: For each bootstrap replicate:
\enumerate{
\item Resample rows using block bootstrap (preserving temporal dependence)
\item Perform PCA on resampled data
\item Apply Procrustes rotation to align with original
\item Record absolute loadings
}
The empirical p-value for each loading is the proportion of bootstrap
loadings exceeding the original in absolute value.

\strong{4. Entropy Calculation}: Shannon entropy of squared loadings indicates
whether explanatory power is concentrated (low entropy) or diffuse (high
entropy). High entropy on PC1 suggests systemic co-movement rather than
differentiated structure.
}
\section{Interpretation in Signal Analysis}{

\itemize{
\item \strong{High PC1 entropy}: "Maximum entropy systemic stochasticity" -
the dominant factor captures undifferentiated movement, suggesting
noise rather than latent structure.
\item \strong{Low PC1 entropy}: "Differentiated latent structure" - specific
variables dominate, indicating meaningful groupings.
\item \strong{Significant loadings}: Variables with p < alpha after bootstrap
testing reliably load on that component.
}
}

\examples{
set.seed(123)
n <- 100
p <- 10
X <- matrix(rnorm(n * p), ncol = p)
colnames(X) <- paste0("V", 1:p)
result <- pca_bootstrap(X, n_components = 3, n_boot = 50)
print(result$summary_by_component)

}
\references{
Jolliffe, I. T. (2002). Principal Component Analysis (2nd ed.). Springer.

Kaiser, H. F. (1958). The varimax criterion for analytic rotation in factor
analysis. Psychometrika, 23(3), 187-200.
}
