% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/assist.R
\name{DAGassist}
\alias{DAGassist}
\title{Generate a (console/LaTeX/word/excel/txt) report classifying nodes and comparing models}
\usage{
DAGassist(
  dag,
  formula = NULL,
  data = NULL,
  exposure,
  outcome,
  engine = stats::lm,
  labels = NULL,
  verbose = TRUE,
  type = c("console", "latex", "word", "docx", "excel", "xlsx", "text", "txt", "dwplot",
    "dotwhisker"),
  show = c("all", "roles", "models"),
  out = NULL,
  imply = FALSE,
  eval_all = FALSE,
  exclude = NULL,
  omit_intercept = TRUE,
  omit_factors = TRUE,
  bivariate = FALSE,
  engine_args = list()
)
}
\arguments{
\item{dag}{A \strong{dagitty} object (see \code{\link[dagitty:dagitty]{dagitty::dagitty()}}).}

\item{formula}{Either (a) a standard model formula \code{Y ~ X + ...}, or
(b) a single \strong{engine call} such as \code{feols(Y ~ X + Z | fe, data = df, ...)}.
When an engine call is provided, \code{engine}, \code{data}, and extra arguments are
automatically extracted from the call.}

\item{data}{A \code{data.frame} (or compatible, e.g. tibble). Optional if supplied
via the engine call in \code{formula}.}

\item{exposure}{Optional character scalar; if missing/empty, inferred from the
DAG (must be unique).}

\item{outcome}{Optional character scalar; if missing/empty, inferred from the
DAG (must be unique).}

\item{engine}{Modeling function, default \link[stats:lm]{stats::lm}. Ignored if \code{formula}
is a single engine call (in that case the function is taken from the call).}

\item{labels}{Optional variable labels (named character vector or data.frame).}

\item{verbose}{Logical (default \code{TRUE}). Controls verbosity in the console
printer (formulas + notes).}

\item{type}{Output type. One of
\code{"console"} (default), \code{"latex"}/\code{"docx"}/\code{"word"},
\code{"excel"}/\code{"xlsx"}, \code{"text"}/\code{"txt"},
or the plotting types \code{"dwplot"}/\code{"dotwhisker"}.
For \code{type = "latex"}, if no \verb{out=} is supplied, a LaTeX fragment is printed
to the console instead of being written to disk.}

\item{show}{Which sections to include in the output. One of \code{"all"} (default),
\code{"roles"} (only the roles grid), or \code{"models"} (only the model comparison table/plot).
This makes it possible to generate and export just roles or just comparisons.}

\item{out}{Output file path for the non-console types:
\itemize{
\item \code{type="latex"}: a \strong{LaTeX fragment} written to \code{out} (usually \code{.tex});
when omitted, the fragment is printed to the console.
\item \code{type="text"}/\code{"txt"}: a \strong{plain-text} file written to \code{out};
when omitted, the report is printed to console.
\item \code{type="dotwhisker"}/\code{"dwplot"}: a \strong{image (.png)} file written to \code{out};
when omitted, the plot is rendered within RStudio.
\item \code{type="docx"}/\code{"word"}: a \strong{Word (.docx)} file written to \code{out}.
\item \code{type="excel"}/\code{"xlsx"}: an \strong{Excel (.xlsx)} file written to \code{out}.
Ignored for \code{type="console"}.
}}

\item{imply}{Logical; default \code{FALSE}. Specifies \strong{evaluation scope.}
\itemize{
\item If \code{FALSE} (default): restrict DAG evaluation to variables \strong{named in the formula}
(prune the DAG to exposure, outcome, and RHS terms). Roles/sets/bad-controls are
computed on this pruned graph, and the roles table \strong{only} shows those variables.
Essentially, it fits the DAG to the formula.
\item If \code{TRUE}: evaluate on the \strong{full DAG} and allow DAG-implied controls in the
minimal/canonical sets. The roles table shows all DAG nodes, and the printout
notes any variables added beyond your RHS. Essentially, it fits the formula to the DAG.
}}

\item{eval_all}{Logical; default \code{FALSE}.  When \code{TRUE}, keep \strong{all original RHS terms}
that are not in the DAG (e.g., fixed effects, interactions, splines,
convenience covariates) in the minimal and canonical formulas.
When \code{FALSE} (default), RHS terms not present as DAG nodes are dropped
from those derived formulas.}

\item{exclude}{Optional character vector to remove neutral controls from the canonical set.
Recognized values are \code{"nct"} (drop \emph{neutral-on-treatment} controls) and
\code{"nco"} (drop \emph{neutral-on-outcome} controls). You can supply one or both,
e.g. \code{exclude = c("nco", "nct")}; each requested variant is fitted and shown
as a separate "Canon. (-...)" column in the console/model exports.}

\item{omit_intercept}{Logical; drop intercept rows from the model comparison display (default \code{TRUE}).}

\item{omit_factors}{Logical; drop factor-level rows from the model comparison display (default \code{TRUE}).
This parameter only suppresses factor \strong{output}--they are still included in the regression.}

\item{bivariate}{Logical; if \code{TRUE}, include a bivariate (exposure-only) specification
in the comparison table \strong{in addition} to the user's original and DAG-derived models.}

\item{engine_args}{Named list of extra arguments forwarded to \code{engine(...)}.
If \code{formula} is an engine call, arguments from the call are merged with
\code{engine_args} (call values take precedence).}
}
\value{
An object of class \code{"DAGassist_report"}, invisibly for file and plot
outputs, and printed for \code{type="console"}. The list contains:
\itemize{
\item \code{validation} - result from \code{validate_spec(...)} which verifies acyclicity and X/Y declarations.
\item \code{roles} - raw roles data.frame from \code{classify_nodes(...)} (logic columns).
\item \code{roles_display} - roles grid after labeling/renaming for exporters.
\item \code{bad_in_user} - variables in the user's RHS that are \code{MED}/\code{COL}/\code{dOut}/\code{dMed}/\code{dCol}.
\item \code{controls_minimal} - (legacy) one minimal set (character vector).
\item \code{controls_minimal_all} - list of all minimal sets (character vectors).
\item \code{controls_canonical} - canonical set (character vector; may be empty).
\item \code{controls_canonical_excl} - named list of filtered canonical sets
(e.g. \verb{$nco}, \verb{$nct}) when \code{exclude} is used.
\item \code{formulas} - list with \code{original}, \code{minimal}, \code{minimal_list}, \code{canonical},
and any filtered canonical formulas.
\item \code{models} - list with fitted models \code{original}, \code{minimal}, \code{minimal_list},
\code{canonical}, and any filtered canonical models.
\item \code{verbose}, \code{imply} - flags as provided.
}
}
\description{
\code{DAGassist()} validates a DAG + model specification, classifies node roles,
builds minimal and canonical adjustment sets, fits comparable models, and
renders a compact report in several formats (console, LaTeX fragment, DOCX,
XLSX, plain text). It also supports passing a \strong{single engine call} (e.g.
\code{feols(Y ~ X + Z | fe, data = df)}) instead of a plain formula.
}
\details{
In addition to tabular export formats, you can create a dot-whisker plot
(via \code{type = "dwplot"} or \code{type = "dotwhisker"}) for the model comparison.

\strong{Engine-call parsing.} If \code{formula} is a call (e.g., \code{feols(Y ~ X | fe, data=df)}),
DAGassist extracts the engine function, formula, data argument, and any additional
engine arguments directly from that call; these are merged with \code{engine}/\code{engine_args}
you pass explicitly (call arguments win).

\strong{fixest tails.} For engines like \strong{fixest} that use \code{|} to denote FE/IV parts,
DAGassist preserves any \verb{| ...} tail when constructing minimal/canonical formulas
(e.g., \code{Y ~ X + controls | fe | iv(...)}).

\strong{Roles grid.} The roles table displays short headers:
\itemize{
\item \code{Exp.} (exposure),
\item \code{Out.} (outcome),
\item \code{CON} (confounder),
\item \code{MED} (mediator),
\item \code{COL} (collider),
\item \code{dOut} (descendant of \code{Y}),
\item \code{dMed} (descendant of any mediator),
\item \code{dCol} (descendant of any collider),
\item \code{dConfOn} (descendant of a confounder \strong{on} a back-door path),
\item \code{dConfOff} (descendant of a confounder \strong{off} a back-door path),
\item \code{NCT} (neutral control on treatment),
\item \code{NCO} (neutral control on outcome).
These extra flags are used to (i) warn about bad controls, and (ii) build
filtered canonical sets such as “Canonical (–NCO)” for export.
}

\strong{Bad controls.} For total-effect estimation, DAGassist flags as \verb{bad controls}
any variables that are \code{MED}, \code{COL}, \code{dOut}, \code{dMed}, or \code{dCol}. These are warned in
the console and omitted from the model-comparison table. Valid confounders (pre-treatment)
are eligible for minimal/canonical adjustment sets.

\strong{Output types.}
\itemize{
\item \code{console} prints roles, adjustment sets, formulas (if \code{verbose}), and a compact model comparison
(using \code{{modelsummary}} if available, falling back gracefully otherwise).
\item \code{latex} writes or prints a \strong{LaTeX fragment} you can \verb{\\\\input\{\}} into a paper —
it uses \code{tabularray} long tables and will include any requested Canon. (-NCO / -NCT) variants.
\item \code{docx}/\code{word} writes a \strong{Word} doc (respects \code{options(DAGassist.ref_docx=...)} if set).
\item \code{excel}/\code{xlsx} writes an \strong{Excel} workbook with tidy tables.
\item \code{text}/\code{txt} writes a \strong{plain-text} report for logs/notes.
\item \code{dwplot}/\code{dotwhisker} produces a dot-whisker visualization of the fitted models.
}

\strong{Dependencies.} Core requires \code{{dagitty}}. Optional enhancements: \code{{modelsummary}}
(pretty tables), \code{{broom}} (fallback tidying), \code{{rmarkdown}} + \strong{pandoc} (DOCX),
\code{{writexl}} (XLSX), \code{{dotwhisker}}/\code{{ggplot2}} for plotting.
}
\section{Interpreting the output}{

See the vignette articles for worked examples on generating roles-only, models-only,
and LaTeX/Word/Excel reports.

\strong{Model Comparison:}
\itemize{
\item \strong{Minimal} - the smallest adjustment set that blocks all back-door paths
(confounders only).
\item \strong{Canonical} - the largest permissible set: includes all controls that are not
\code{MED}, \code{COL}, \code{dOut}, \code{dMed}, or \code{dCol}.
}
}

\section{Errors and edge cases}{

\itemize{
\item If exposure/outcome cannot be inferred uniquely, the function stops with a clear message.
\item Fitting errors (e.g., FE collinearity) are captured and displayed in comparisons
without aborting the whole pipeline.
}
}

\examples{
\dontshow{if (requireNamespace("dagitty", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\dontshow{set.seed(1)}
\dontshow{
# Build the DAG directly with dagitty
g <- dagitty::dagitty("dag { Z -> X; X -> M; X -> Y; M -> Y; Z -> Y; A -> Y; B -> Y; X -> C; Y -> C }")
dagitty::exposures(g) <- "X"; dagitty::outcomes(g) <- "Y"

n <- 150
A <- rnorm(n); B <- rnorm(n); Z <- rnorm(n)
X <- 0.8*Z + rnorm(n)
M <- 0.9*X + rnorm(n)
Y <- 0.7*X + 0.6*M + 0.3*Z + 0.2*A - 0.1*B + rnorm(n)
C <- 0.5*X + 0.4*Y + rnorm(n)
df <- data.frame(A,B,Z,X,M,Y,C)
}
# generate a console DAGassist report
DAGassist(dag = g, 
          formula = lm(Y ~ X + Z + C + M, data = df))

# generate a LaTeX DAGassist report in console
DAGassist(dag = g, 
          formula = lm(Y ~ X + Z + C + M, data = df),
          type = "latex")

# generate just the roles table in the console
DAGassist(dag = g, 
          show = "roles")
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=print.DAGassist_report]{print.DAGassist_report()}} for the console printer, and the helper
exporters in \verb{report_*} modules.
}
