% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vtrack.R
\name{gvtrack.filter}
\alias{gvtrack.filter}
\title{Attach or clear a genomic mask filter on a virtual track}
\usage{
gvtrack.filter(vtrack = NULL, filter = NULL)
}
\arguments{
\item{vtrack}{virtual track name}

\item{filter}{genomic mask to apply. Can be:
\itemize{
  \item A data.frame with columns 'chrom', 'start', 'end' (intervals to mask)
  \item A character string naming an intervals set
  \item A character string naming a track (must be intervals-type track)
  \item A list of any combination of the above (all will be unified)
  \item NULL to clear the filter
}}
}
\value{
None (invisibly).
}
\description{
Attaches or clears a genomic mask filter on a virtual track. When a filter is attached,
the virtual track function is evaluated only over the unmasked regions (i.e., regions
not covered by the filter intervals).
}
\details{
The filter defines regions to \emph{exclude} from virtual track evaluation.
The virtual track function will be evaluated only on the complement of the filter.
Once a filter is attached to a virtual track, it applies to \strong{all subsequent extractions}
of that virtual track until explicitly cleared with \code{filter = NULL}.

\strong{Order of Operations:}

Filters are applied \emph{after} iterator modifiers (sshift/eshift/dim). The order is:
\enumerate{
  \item Apply iterator modifiers (gvtrack.iterator with sshift/eshift)
  \item Subtract mask from the modified intervals
  \item Evaluate virtual track function over unmasked regions
}

\strong{Semantics by function type:}
\itemize{
  \item \strong{Aggregations (avg/sum/min/max/stddev/quantile):} Length-weighted over unmasked regions
  \item \strong{coverage:} Returns (covered length in unmasked region) / (total unmasked length)
  \item \strong{distance/distance.center:} Unaffected by mask (pure geometry)
  \item \strong{PWM/kmer:} Masked bases act as hard boundaries; matches cannot span masked regions.
        \strong{Important:} When \code{extend=TRUE} (the default), motifs at the boundaries of unmasked
        segments can use bases from the adjacent masked regions to complete the motif scoring.
        For example, if a 4bp motif starts at position 1998 in an unmasked region that ends at 2000,
        and positions 2000-2002 are masked, the motif will still be scored using the masked bases.
        In other words, motif matches \emph{starting positions} must be in unmasked regions,
        but the motif sequence itself can extend into masked regions when \code{extend=TRUE}.
        Set \code{extend=FALSE} to prevent any use of masked bases in scoring.
}

\strong{Completely Masked Intervals:}
If an entire iterator interval is masked, the function returns \code{NA} (not 0).
}
\examples{
\dontshow{
options(gmax.processes = 2)
}

gdb.init_examples()

## Basic usage: Excluding specific regions
gvtrack.create("vtrack1", "dense_track", func = "avg")

# Create intervals to mask (e.g., repetitive regions)
repeats <- gintervals(c(1, 1), c(100, 500), c(200, 600))

# Attach filter - track will be evaluated excluding these regions
gvtrack.filter("vtrack1", filter = repeats)

# Extract values - masked regions are excluded from calculation
result_filtered <- gextract("vtrack1", gintervals(1, 0, 1000))

# Check filter info
gvtrack.info("vtrack1")

# Clear the filter and compare
gvtrack.filter("vtrack1", filter = NULL)
result_unfiltered <- gextract("vtrack1", gintervals(1, 0, 1000))

## Using multiple filter sources (combined automatically)
centromeres <- gintervals(1, 10000, 15000)
telomeres <- gintervals(1, 0, 1000)
combined_mask <- list(repeats, centromeres, telomeres)

gvtrack.filter("vtrack1", filter = combined_mask)
result_multi_filter <- gextract("vtrack1", gintervals(1, 0, 20000))

## Filters work with iterator modifiers
gvtrack.create("vtrack2", "dense_track", func = "sum")
gvtrack.filter("vtrack2", filter = repeats)
gvtrack.iterator("vtrack2", sshift = -50, eshift = 50)

# Iterator shifts applied first, then mask subtracted
result_shifted <- gextract("vtrack2", gintervals(1, 1000, 2000), iterator = 100)

}
\seealso{
\code{\link{gvtrack.create}}, \code{\link{gvtrack.iterator}}, \code{\link{gvtrack.info}}
}
\keyword{~filter}
\keyword{~virtual}
