#' @title Solve linear system
#' @description This function classifies and solves the given linear system.
#' @param A The augmented matrix of a linear system.
#' @param tol A tolerance parameter, as a non-negative number.\cr
#'            By default, \code{tol=100*.Machine$double.eps}.
#' @return This function returns two outputs: \code{solution} and \code{flag}.
#' If the introduced linear system is inconsistent: \code{flag=-1} and \code{solution=Inf}.
#' If it is consistent and has infinitely many solutions: \code{flag=0} and \code{solution} returns one of the solutions, as a vector.
#' If it is consistent and has a unique solution: \code{flag=1} and \code{solution} returns the unique solution, as a vector.
#' @examples
#' # Consistent and determinate system:
#' solvels(matrix(c(1,1,1,6,2,-1,1,3,-1,-1,1,0), byrow=TRUE, nrow = 3, ncol = 4))
#' # Consistent and indeterminate system:
#' solvels(matrix(c(1,1,-3,0,2,-1,-3,3,4,1,-9,3), byrow=TRUE, nrow = 3, ncol = 4))
#' # Inconsistent system:
#' solvels(matrix(c(-2,1,1,1,1,-2,1,1,1,1,-2,1), byrow=TRUE, nrow = 3, ncol = 4))
#' @export

solvels <- function(A, tol = 100*.Machine$double.eps) {
  #solvels Classifies and solves the linear system given by the augmented matrix A

  # INPUT
  # A Augmented matrix of the linear system
  # tol Tolerance level. Default tol=100*eps
  #
  # OUTPUT
  # x A solution of the linear system
  # flag = -1 . Inconsistent system  (x=Inf)
  # flag = 0  . Consistent system with infinitely many solutions
  # flag = 1  . Consistent system with a unique solution
  # A <- matrix(c(1,1,1,6,2,-1,1,3,-1,-1,1,0),byrow=TRUE, nrow = 3, ncol = 4) SCD
  # A <- matrix(c(1,1,-3,0,2,-1,-3,3,4,1,-9,3),byrow=TRUE, nrow = 3, ncol = 4) SCI
  # A <- matrix(c(-2,1,1,1,1,-2,1,1,1,1,-2,1),byrow=TRUE, nrow = 3, ncol = 4) SI
  # A será una matriz mx(n+1)
  # El sistema que resolveremos es A(:,1:n)*x=A(:,n+1)
  # Es decir A es la matriz ampliada del sistema.
  # Primero triangulamos A (método de Gauss)
  # mediante una matriz cuadrada T de orden (n+1)x(n+1).
  # Luego clasificamos y resolvemos escalonadamente.
  #
  #  Si no nos dan la tolerancia, fijar tol = 100 * .Machine$double.eps

  # Número de incógnitas: n
  n <- ncol(A) - 1

  # Triangulamos la matriz ampliada A
  T <- triangularup(A, tol)$SUT

  # Recordemos que T es cuadrada (n+1)x(n+1) y con 0 o 1 en la diagonal.
  # Luego la última fila será 0 ... 0 | T(n+1,n+1)
  # Clasificamos el sistema
  # disp('Rango de la matriz del sistema: '), rank(T(:,1:n))
  # disp('Rango de la matriz ampliada: '), rank(T)
  # Primero miro los casos de sistema incompatible
  # Ultima ecuacion: 0 ... 0 | 1
  flag <- 1
  x <- NULL

  # Caso de sistema incompatible
  if (abs(T[n + 1, n + 1]) > tol) {
    flag <- -1 # Sistema incompatible
    x <- Inf
  }

  # Las otras ecuaciones
  ii <- n + 1
  while (flag == 1 && ii >= 2) {
    ii <- ii - 1
    if (abs(T[ii, ii]) < tol && abs(T[ii, n + 1]) > tol) {
      flag <- -1 # Sistema incompatible
      x <- Inf
    }
  }

  # En caso contrario sistema compatible
  # Comprobamos si el sistema es compatible indeterminado, es decir,
  # si tiene algún 0 en la diagonal
  ii <- n + 1
  while (flag == 1 && ii >= 2) {
    ii <- ii - 1
    if (abs(T[ii, ii]) < tol) { # Un 0 en la diagonal
      flag <- 0 # Sistema compatible indeterminado
    }
  }
  # Una solución del sistema

  # Si flag=1 o flag=0: Sistema compatible
  if (flag != -1) {
    # Solucion escalonada
    x <- numeric(n) # Inicializamos el vector solución en ceros
    # En caso de sistema compatible indeterminado daremos una solución tal
    # que todas las coordenadas ii que no tengan un 1 en la diagonal de T
    # (T(ii,ii)=0) serán nulas.
    for (ii in seq(n, 1)) { # Desde la última ecuación "válida" hacia arriba
      if (abs(T[ii, ii]) > tol) { # i el coeficiente en la diagonal no es nulo
        # Recordemos que en el proceso de triangulación
        # pusimos 0 o 1 en la diagonal. Luego no tengo que dividir entre T(ii,ii)
        # Despejando, x(ii) es el término independiente de la fila menos la
        # suma de los valores ya calculados por sus correspondientes
        # coeficientes
        aux <- T[ii, n + 1]
        if (ii == n){x[ii] <- aux}
        else
        {for (jj in seq(ii + 1, n)) {
          aux <- aux - T[ii, jj] * x[jj]
        }
        x[ii] <- aux}
      }
    } # fin del bucle en las filas del sistema
  } #fin del bucle flag

  return(list(solution = x, flag = flag))
}
