#' Calculates DKI (v2)
#'
#' @description
#' [DKI2()] calculate a salinity-normalised version of the Danish quality
#' index (DKI) [(Carstensen et al., 2014)](#references)
#'
#' The *DKI* index is based on AMBI and can only be calculated after first calculating
#' *AMBI*, the AZTI Marine Biotic Index, and *H'*, the Shannon diversity index.
#' Both indices are included in output from the function [AMBI()].
#'
#' This function uses linear relationships between salinity and limits for `AMBI`
#' and `Hdash` to normalise the index. This is done to account for expected
#' lower species diversity in regions with lower salinity.
#'
#' Since the index is normalised to salinity, the function also requires
#' measured or estimated salinity `psal` as an argument.
#'
#'#' @references
#' Carstensen, J., Krause-Jensen, D., Josefson, A. (2014). "Development and testing of tools for intercalibration of phytoplankton, macrovegetation and benthic fauna in Danish coastal areas." Aarhus University, DCE – Danish Centre for Environment and Energy, 85 pp. _Scientific Report from DCE – Danish Centre for Environment and Energy_ No. 93.
#' <https://dce2.au.dk/pub/SR93.pdf>
#'
#' @details
#' The [AMBI()] and [Hdash()] functions take a dataframe of observations as an
#' argument. The *DKI* functions, [DKI2()] and [DKI()], do *not* take a dataframe
#' as an argument. Instead they take values of the input parameters, either
#' single values or as vectors.
#'
#' To calculate DKI for a dataframe of `AMBI` values, it could be called from
#' e.g. within a [dplyr::mutate()] function call. See the examples below.
#'
#' @seealso
#' * [DKI()] calculate DKI using the original method
#' * [AMBI_sal()] minimum AMBI for normalisation _= f(salinity)_
#' * [H_sal()] maximum H' for normalisation _= f(salinity)_
#'
#'
#' @param AMBI        AMBI, the AZTI Marine Biotic Index, calculated using [AMBI()]
#' @param H           H', the Shannon diversity index, calculated using [Hdash()]
#' @param N           number of individuals - generated by both [AMBI()] and [Hdash()]
#' @param psal        salinity
#'
#' @return
#' `DKI` index value
#'
#' @examples
#'
#' # Simple example
#'
#' DKI2(AMBI = 1.61, H = 2.36, N = 25, psal = 21.4)
#'
#'
#' # ------ Example workflow for calculating DKI (v2) from species counts ----
#'
#' # calculate AMBI index
#' dfAMBI <- AMBI(test_data, by = c("station"), var_rep = "replicate")[["AMBI"]]
#'
#' # show AMBI results
#' dfAMBI
#'
#' # add salinity values - these are realistic but invented values
#' dfAMBI <- dplyr::mutate(dfAMBI, psal=ifelse(station == 1, 21.3, 26.5))
#'
#' # calculate DKI from AMBI results
#' dfAMBI <- dplyr::mutate(dfAMBI, DKI=DKI2(AMBI, H, N, psal))
#'
#' @export

DKI2 <- function(AMBI, H, N, psal){

  AMBI_sal <- AMBI_sal(psal)
  H_sal <- H_sal(psal)
  term_AMBI <- (1 - ((AMBI - AMBI_sal) / 7))
  term_H <- H/H_sal
  term_H <- ifelse(term_H > 1, 1, term_H)
  DKI2 <- 0.5*(term_AMBI + term_H) * (1 - (1/N))
  return(DKI2)
}

