test_that("score_anomaly works with iforest method", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15),
    cost = rnorm(100, 10000, 5000)
  )
  
  result <- score_anomaly(data, method = "iforest", contamination = 0.05)
  
  expect_true(is.data.frame(result))
  expect_true("anomaly_score" %in% names(result))
  expect_true(all(result$anomaly_score >= 0 & result$anomaly_score <= 1))
  expect_true(nrow(result) == 100)
  expect_equal(attr(result, "method"), "iforest")
})

test_that("score_anomaly works with lof method", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15),
    cost = rnorm(100, 10000, 5000)
  )
  
  result <- score_anomaly(data, method = "lof", contamination = 0.05)
  
  expect_true(is.data.frame(result))
  expect_true("anomaly_score" %in% names(result))
  expect_true(all(result$anomaly_score >= 0 & result$anomaly_score <= 1))
  expect_equal(attr(result, "method"), "lof")
})

test_that("score_anomaly calculates benchmarking metrics with ground truth", {
  set.seed(123)
  data <- data.frame(
    patient_id = 1:200,
    age = rnorm(200, 50, 15),
    cost = rnorm(200, 10000, 5000),
    is_error = sample(c(0, 1), 200, replace = TRUE, prob = c(0.95, 0.05))
  )
  
  # Introduce errors
  error_idx <- which(data$is_error == 1)
  data$cost[error_idx] <- data$cost[error_idx] * 10
  
  result <- score_anomaly(data, method = "iforest", 
                          contamination = 0.05,
                          ground_truth_col = "is_error")
  
  metrics <- attr(result, "benchmark_metrics")
  
  expect_false(is.null(metrics))
  expect_true("auc_roc" %in% names(metrics))
  expect_true("auc_pr" %in% names(metrics))
  expect_true("top_k_recall" %in% names(metrics))
  expect_true(is.numeric(metrics$auc_roc))
  expect_true(is.numeric(metrics$auc_pr))
  expect_true(metrics$auc_roc >= 0 && metrics$auc_roc <= 1)
  expect_true(metrics$auc_pr >= 0 && metrics$auc_pr <= 1)
})

test_that("score_anomaly handles different contamination rates", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15),
    cost = rnorm(100, 10000, 5000)
  )
  
  result_01 <- score_anomaly(data, contamination = 0.01)
  result_10 <- score_anomaly(data, contamination = 0.10)
  
  expect_true(is.data.frame(result_01))
  expect_true(is.data.frame(result_10))
  expect_true(all(result_01$anomaly_score >= 0 & result_01$anomaly_score <= 1))
  expect_true(all(result_10$anomaly_score >= 0 & result_10$anomaly_score <= 1))
})

test_that("score_anomaly errors on invalid inputs", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15)
  )
  
  expect_error(score_anomaly("not a data frame"))
  expect_error(score_anomaly(data, method = "invalid_method"))
  expect_error(score_anomaly(data, contamination = -1))
  expect_error(score_anomaly(data, contamination = 1.5))
  expect_error(score_anomaly(data, ground_truth_col = "nonexistent"))
})

test_that("score_anomaly handles ground truth as logical", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15),
    cost = rnorm(100, 10000, 5000),
    is_error = sample(c(TRUE, FALSE), 100, replace = TRUE, prob = c(0.05, 0.95))
  )
  
  result <- score_anomaly(data, ground_truth_col = "is_error")
  
  metrics <- attr(result, "benchmark_metrics")
  expect_false(is.null(metrics))
})

test_that("score_anomaly handles ground truth as factor", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15),
    cost = rnorm(100, 10000, 5000),
    is_error = factor(sample(c(0, 1), 100, replace = TRUE, prob = c(0.95, 0.05)))
  )
  
  result <- score_anomaly(data, ground_truth_col = "is_error")
  
  metrics <- attr(result, "benchmark_metrics")
  expect_false(is.null(metrics))
})

test_that("score_anomaly preserves original data columns", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15),
    cost = rnorm(100, 10000, 5000),
    gender = sample(c("M", "F"), 100, replace = TRUE)
  )
  
  result <- score_anomaly(data)
  
  expect_true(all(c("patient_id", "age", "cost", "gender", "anomaly_score") %in% names(result)))
  expect_equal(ncol(result), ncol(data) + 1)
})

test_that("score_anomaly handles small datasets", {
  data <- data.frame(
    patient_id = 1:10,
    age = rnorm(10, 50, 15),
    cost = rnorm(10, 10000, 5000)
  )
  
  result <- score_anomaly(data, method = "iforest")
  
  expect_true(is.data.frame(result))
  expect_true("anomaly_score" %in% names(result))
})

test_that("score_anomaly passes additional arguments to algorithms", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15),
    cost = rnorm(100, 10000, 5000)
  )
  
  # Test with custom ntrees for iforest
  result <- score_anomaly(data, method = "iforest", ntrees = 50)
  
  expect_true(is.data.frame(result))
  expect_true("anomaly_score" %in% names(result))
  
  # Test with custom minPts for lof
  result_lof <- score_anomaly(data, method = "lof", minPts = 5)
  
  expect_true(is.data.frame(result_lof))
  expect_true("anomaly_score" %in% names(result_lof))
})

