% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bcajack.R
\name{bcajack}
\alias{bcajack}
\title{Nonparametric bias-corrected and accelerated bootstrap
confidence limits}
\usage{
bcajack(
  x,
  B,
  func,
  ...,
  m = nrow(x),
  mr = 5,
  K = 2,
  J = 10,
  alpha = c(0.025, 0.05, 0.1, 0.16),
  verbose = TRUE
)
}
\arguments{
\item{x}{an \eqn{n \times p} data matrix, rows are observed
\eqn{p}-vectors, assumed to be independently sampled from
target population. If \eqn{p} is 1 then \code{x} can be a vector.}

\item{B}{number of bootstrap replications. It can also be a vector
of \code{B} bootstrap replications of the estimated parameter of
interest, computed separately.}

\item{func}{function \eqn{\hat{\theta}=func(x)} computing estimate of the
parameter of interest; \eqn{func(x)} should return a real value
for any \eqn{n^\prime \times p} matrix \eqn{x^\prime},
\eqn{n^\prime} not necessarily equal to \eqn{n}}

\item{...}{additional arguments for \code{func}.}

\item{m}{an integer less than or equal to \eqn{n}; the routine
collects the \eqn{n} rows of \code{x} into \code{m} groups to speed up
the jackknife calculations for estimating the acceleration
value \eqn{a}; typically \code{m} is 20 or 40 and does not have to
exactly divide \eqn{n}. However, warnings will be shown.}

\item{mr}{if \eqn{m < n} then \code{mr} repetions of the randomly
grouped jackknife calculations are averaged.}

\item{K}{a non-negative integer. If \code{K} > 0, bcajack also returns
estimates of \emph{internal standard error}, that is, of the
variability due to stopping at \code{B} bootstrap replications
rather than going on to infinity. These are obtained from a
second type of jackknifing, taking an average of \code{K} separate
jackknife estimates, each randomly splitting the \code{B} bootstrap
replications into \code{J} groups.}

\item{J}{the number of groups into which the bootstrap replications
are split}

\item{alpha}{percentiles desired for the bca confidence limits. One
only needs to provide \code{alpha} values below 0.5; the upper
limits are automatically computed}

\item{verbose}{logical for verbose progress messages}
}
\value{
a named list of several items
\itemize{
\item \strong{lims} : first column shows the estimated bca confidence limits
at the requested alpha percentiles. These can be compared with
the standard limits \eqn{\hat{\theta} +
    \hat{\sigma}z_{\alpha}}, third column. The second column
\code{jacksd} gives the internal standard errors for the bca limits,
quite small in the example. Column 4, \code{pct}, gives the
percentiles of the ordered B bootstrap replications
corresponding to the bca limits, eg the 897th largest
replication equalling the .975 bca limit .557.
\item \strong{stats} : top line of stats shows 5 estimates: theta is
\eqn{f(x)}, original point estimate of the parameter of
interest; \code{sdboot} is its bootstrap estimate of standard error;
\code{z0} is the bca bias correction value, in this case quite
negative; \code{a} is the \emph{acceleration}, a component of the bca
limits (nearly zero here); \code{sdjack} is the jackknife estimate
of standard error for theta. Bottom line gives the internal
standard errors for the five quantities above. This is
substantial for \code{z0} above.
\item \strong{B.mean} : bootstrap sample size B, and the mean of the B
bootstrap replications \eqn{\hat{\theta^*}}
\item \strong{ustats} : The bias-corrected estimator \code{2 * t0 - mean(tt)},
and an estimate \code{sdu} of its sampling error
\item \strong{seed} : The random number state for reproducibility
}
}
\description{
This routine computes nonparametric confidence
intervals for bootstrap estimates. For reproducibility, save or
set the random number state before calling this routine.
}
\details{
Bootstrap confidence intervals depend on three elements:
\itemize{
\item the cdf of the \eqn{B} bootstrap replications \eqn{t_i^*}, \eqn{i=1\ldots B}
\item the bias-correction number \eqn{z_0=\Phi(\sum_i^B I(t_i^* < t_0) / B )}
where \eqn{t_0=f(x)} is the original estimate
\item the acceleration number \eqn{a} that measures the rate of
change in \eqn{\sigma_{t_0}} as \eqn{x}, the data changes.
}

The first two of these depend only on the bootstrap distribution,
and not how it is generated: parametrically or
non-parametrically. Program bcajack can be used in a hybrid fashion
in which the vector \code{tt} of B bootstrap replications is first
generated from a parametric model.

So, in the diabetes example below, we might first draw bootstrap
samples \eqn{y^* \sim N(X\hat{\beta}, \hat{\sigma}^2 I)} where
\eqn{\hat{\beta}} and \eqn{\hat{\sigma}} were obtained from
\code{lm(y~X)}; each \eqn{y^*} would then provide a bootstrap
replication \code{tstar = rfun(cbind(X, ystar))}.  Then we could get bca
intervals from \verb{bcajack(Xy, tt, rfun ....)} with \code{tt},
the vector of B \code{tstar} values. The only difference from a full
parametric bca analysis would lie in the nonparametric estimation
of \eqn{a}, often a negligible error.
}
\examples{
data(diabetes, package = "bcaboot")
Xy <- cbind(diabetes$x, diabetes$y)
rfun <- function(Xy) {
  y <- Xy[, 11]
  X <- Xy[, 1:10]
  summary(lm(y~X) )$adj.r.squared
}
set.seed(1234)
## n = 442 = 34 * 13
bcajack(x = Xy, B = 1000, func = rfun, m = 34, verbose = FALSE)
}
\references{
DiCiccio T and Efron B (1996). Bootstrap confidence
intervals. Statistical Science 11, 189-228

Efron B (1987). Better bootstrap confidence
intervals. JASA 82 171-200

B. Efron and B. Narasimhan. Automatic Construction of
Bootstrap Confidence Intervals, 2018.
}
