% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_group_trajectories_lines_3d.R
\name{plot_group_trajectories_lines_3d}
\alias{plot_group_trajectories_lines_3d}
\title{Plot 3D trajectories as variable-width lines}
\usage{
plot_group_trajectories_lines_3d(
  traj_data,
  traj_filtered,
  width_range_hi = c(4, 12),
  width_range_lo = c(1.2, 3),
  use_raw_papers = TRUE,
  connect_only_existing_edges = TRUE,
  show_labels = TRUE,
  show_only_highlighted = FALSE,
  label_size = 18,
  hover_font_size = 12,
  lowlight_width = 1,
  lowlight_alpha = 0.9,
  lowlight_color = "#9AA5B1"
)
}
\arguments{
\item{traj_data}{List containing trajectory data generated by
\code{detect_main_trajectories()} with components:
\itemize{
\item \code{graph}: igraph object containing nodes and edges across years
\item \code{trajectories}: tibble of all candidate trajectories (traj_id + nodes list)
}}

\item{traj_filtered}{Filtered trajectories tibble from \code{filter_trajectories()}
containing the subset to emphasize. Must contain columns:
\itemize{
\item \code{traj_id}: trajectory identifiers
\item \code{nodes}: list of character vectors (ordered by time or orderable)
}}

\item{width_range_hi}{Width range for highlighted trajectory segments
(default: c(4, 12)). Segment widths scale with cumulative paper counts.}

\item{width_range_lo}{Baseline width range used to compute constant lowlight
width (default: c(1.2, 3)). The mean of this range determines lowlight width.}

\item{use_raw_papers}{Whether to use raw paper counts for z-axis scaling
(default: TRUE). If TRUE, uses raw \code{quantity_papers}; if FALSE, uses
weighted size: \code{quantity_papers * prop_tracked_intra_group}.}

\item{connect_only_existing_edges}{Whether to draw only edges that exist in
the graph (default: TRUE). If FALSE, draws all consecutive node pairs in
trajectories regardless of graph edges.}

\item{show_labels}{Whether to add end-of-trajectory labels inside the 3D plot
(default: TRUE)}

\item{show_only_highlighted}{Whether to show only highlighted trajectories
(default: FALSE). If TRUE, hides all background network and lowlight trajectories.}

\item{label_size}{Font size for trajectory end labels (default: 18)}

\item{hover_font_size}{Font size for hover tooltips (default: 12)}

\item{lowlight_width}{Line width for lowlight trajectories and background
network (default: 1)}

\item{lowlight_alpha}{Transparency for lowlight elements (default: 0.9)}

\item{lowlight_color}{Color for lowlight elements (default: "#9AA5B1" - neutral gray)}
}
\value{
A plotly interactive 3D plot object
}
\description{
Creates an interactive 3D plot showing research trajectories with time on the x-axis,
route separation on the y-axis, and cumulative paper counts on the z-axis. Highlighted
trajectories are displayed as growing-thickness lines, with optional background
trajectories and network context in lowlight style.
}
\details{
This function creates an interactive 3D visualization of research trajectories:
\itemize{
\item \strong{X-axis}: Publication year (parsed from vertex names like "y2007g05")
\item \strong{Y-axis}: "Route" (Sugiyama layout coordinate to separate trajectories vertically)
\item \strong{Z-axis}: Cumulative documents (raw or weighted) along each trajectory
}

Key features:
\itemize{
\item \strong{Highlighted trajectories} (\code{traj_filtered}) are colored lines with widths
that grow proportionally to cumulative paper counts
\item \strong{Lowlight trajectories} (when \code{show_only_highlighted = FALSE}) show other
trajectories as constant-width lines
\item \strong{Background network} (when \code{show_only_highlighted = FALSE}) provides
context with thin gray edges
\item \strong{Hover tooltips} show detailed information at each trajectory point
\item \strong{End labels} identify highlighted trajectories (when \code{show_labels = TRUE})
\item \strong{Edge validation} (when \code{connect_only_existing_edges = TRUE}) ensures only
actual graph edges are drawn
}

The function uses a Sugiyama layout for the y-axis coordinates and cumulative
sums of paper counts for the z-axis values. Colors for highlighted trajectories
are assigned using RColorBrewer's Set2 palette (for <=8 trajectories) or a
hue-based palette (for more trajectories).
}
\examples{
\dontrun{
# Detect main trajectories first
traj_data <- detect_main_trajectories(your_graph_data)

# Filter trajectories of interest
filtered_traj <- filter_trajectories(traj_data$trajectories, 
                                     min_papers = 10)

# Create interactive 3D plot
plot_group_trajectories_lines_3d(
  traj_data = traj_data,
  traj_filtered = filtered_traj,
  width_range_hi = c(3, 10),
  use_raw_papers = FALSE,
  show_labels = TRUE
)

# Minimal view with only highlighted trajectories
plot_group_trajectories_lines_3d(
  traj_data = traj_data,
  traj_filtered = filtered_traj,
  show_only_highlighted = TRUE,
  label_size = 16
)
}

}
