% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hmm.R
\name{run_hmm}
\alias{run_hmm}
\title{Hidden Markov Model (HMM) for Path Dependence (Counts I and C)}
\usage{
run_hmm(DT, nstates = 3, seed = NULL)
}
\arguments{
\item{DT}{A \code{data.frame} or \code{data.table} containing at least
the columns \code{I} and \code{C}, interpreted as non-negative count
series observed over time.}

\item{nstates}{Integer; number of latent Markov states to fit in the
HMM (default is 3).}

\item{seed}{Integer or NULL; optional seed for reproducibility.
If NULL (default), no seed is set and results may vary between runs.}
}
\value{
If the optimization succeeds, a list with components:
\itemize{
\item \code{fit}: the fitted \code{"depmix"} model object.
\item \code{states}: integer vector of inferred latent states (one
per time point).
}
If fitting fails (e.g., non-convergence), the function returns
\code{NULL}.
}
\description{
Fits a univariate time-series Hidden Markov Model (HMM) with Poisson
emissions for the count variables \code{I} and \code{C} using
\pkg{depmixS4}. The estimated state sequence is exported and the fit
object is saved to disk.
}
\details{
The model is specified via \code{depmixS4::depmix()} as a multivariate
Poisson HMM with two observed series:
\itemize{
\item \code{I ~ 1}
\item \code{C ~ 1}
}
and \code{nstates} hidden regimes. The function:
\enumerate{
\item Builds a data frame with columns \code{I} and \code{C}.
\item Constructs the HMM with Poisson emission distributions for both
series.
\item Optionally sets a random seed if the \code{seed} argument is provided.
\item Fits the model with \code{fit(mod, verbose = FALSE)} wrapped in
\code{try()} to avoid stopping on optimization failures.
\item If fitting succeeds, extracts the posterior state sequence via
\code{depmixS4::posterior()}.
}

The function assumes that two global character scalars are defined:
\itemize{
\item \code{dir_csv}: directory where the state sequence CSV will
be written.
\item \code{dir_out}: directory where the fitted HMM object RDS will
be saved.
}
A CSV file named \code{"hmm_states.csv"} is written to \code{dir_csv}
with columns \code{t} (time index) and \code{state} (most probable
state). The fitted HMM object is saved as \code{"hmm_fit.rds"} in
\code{dir_out}.
}
\examples{
\donttest{
library(data.table)

# 1. Create dummy data (Only 'I' and 'C' counts are required by this function)
DT <- data.table(
  I = rpois(50, lambda = 4),
  C = rpois(50, lambda = 3)
)

# 2. Define global paths using tempdir() (Fixes CRAN policy)
# run_hmm expects these variables to exist in the global environment
tmp_dir <- tempdir()
dir_csv <- file.path(tmp_dir, "csv")
dir_out <- file.path(tmp_dir, "hmm")

dir.create(dir_csv, showWarnings = FALSE, recursive = TRUE)
dir.create(dir_out, showWarnings = FALSE, recursive = TRUE)

# 3. Run the function
# Using nstates=2 for a faster example check
res_hmm <- run_hmm(DT, nstates = 2)

# Inspect result if successful
if (!is.null(res_hmm)) {
  print(table(res_hmm$states))
}
}

}
