##
# ------------------------------------------------------------------------
#
#  "genMM1(n,lambda,mu)" -- Queuing system 
#
#
# ------------------------------------------------------------------------
##
#' @aliases genMM1
#' @title Generate an M/M/1 Queue Process
#' @description Simulates an M/M/1 queue process.
#'  The M/M/1 queue is a single-server queue with Poisson arrivals and 
#'  exponential service times. Allows to introduce an artificial outlier.
#' @param n An integer value; the length of the process to simulate.
#' @param lambda A positive numeric value; rate parameter for the Poisson 
#' arrival process.
#' @param mu A positive numeric value; rate parameter for the exponential service times.
#' @param out A positive numeric value; the value of an outlier appearing 
#' at time \eqn{floor(n/2)+1}.
#' Default is \code{NULL}, this means that the process is generated without 
#' outlier.
#' @details The M/M/1 waiting time is defined as follows
#' \deqn{X_{t+1} = \max(0, (X_t + U_t - V_t))},
#' where \eqn{U_t} and \eqn{V_t} are exponentially distributed random variables with 
#' parameters \code{lambda} and \code{mu}, respectively. 
#' 
#' @return A numeric vector of length \code{n+1}, representing the waiting time of 
#' customer \eqn{i} arriving at time \eqn{T_i}.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted. 
#' 
#' Asmussen, S. (1987). \emph{Applied Probability and Queues}. Springer N.Y. 
#' @seealso  \code{\link{genETARCH}}, \code{\link{zi_inar_process}}.
#' @keywords "Queueing system"
#' @export
#' @examples 
#' queue = genMM1(100,1,2) 
#' plot(queue, type="l")
#' 
##
genMM1<-function(n,lambda,mu,out=NULL)
{
  X=matrix(0,n+1,1)
  n2=floor(n/2)+1
  X[1]=0;
  U=-log(runif(n))/mu
  DT=-log(runif(n))/lambda
  for (i in 1:n)
  {
    X[i+1]=max(0,(X[i]+U[i]-DT[i]))
    if (is.null(out)==FALSE) {if (i==n2) X[n2+1]=out}
  }
  return(X)
}

##
# ------------------------------------------------------------------------
#
#  "genETARCH(n,alpha1,alpha2,beta)" -- ETAR-ARCH -model
#
#
# ------------------------------------------------------------------------
##
#' @aliases genETARCH
#' @title Generate an Exponential TAR-ARCH Process
#' @description Simulates an Exponential Threshold Autoregressive (ETAR)-ARCH process.
#' @param n An integer value; the length of the process to simulate.
#' @param alpha1 A numeric value; parameter of the process such that \eqn{|alpha1|<1}.
#' @param alpha2 A numeric value; parameter of the process driving the behaviour of the process
#' for small values of \code{x}. \eqn{|alpha1+alpha2|} may be bigger than 1.
#' @param beta A positive numeric value; parameter of the process driving the ARCH effect.
#' @details The ETAR-ARCH process is defined by the equation 
#' \deqn{X_{t+1} = \left(\alpha_1 + \alpha_2 \exp(-X_t^2)\right)X_t + \sqrt{1 + \beta X_t^2} Z_t}, 
#' where \eqn{Z_t} is an i.i.d. error sequence. 
#'
#' @return  A numeric vector of length \code{n+1}, containing the simulated values 
#' of the ETAR-ARCH process.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek)  - submitted. 
#' @seealso \code{\link{genMM1}}, \code{\link{zi_inar_process}}.
#' @keywords "Thresholds models" "Conditional heteroscedasticity"
#' @export
#' @examples 
#' etarch = genETARCH(100,0.3,0.8,0.5) 
#' plot(etarch, type="l")
#' 
##
genETARCH<-function(n,alpha1,alpha2,beta)
{
  #alpha1 should be small #alpha+alpha2 can be bigger  than 1
  #beta is positive
  X=matrix(0,n+1,1)
  X[1]=0
  Z=rnorm(n)
  for (i in 1:n)
  {    X[i+1]=(alpha1+alpha2*exp(-X[i]^2))*X[i]+(1+beta*X[i]^2)^0.5*Z[i]}
  return(X)
}

##
# ------------------------------------------------------------------------
#
#  "zi_inar_process(n, p, alpha, Y = 1, p2, lamb)" -- ZI-INAR(p) - process
#
#
# ------------------------------------------------------------------------
##
#' @aliases zi_inar_process
#' @title Generate a ZI-INAR Process
#' @description Simulates a zero-inflated Poisson distributed (ZI)-INAR(p) process. 
#' @param n A positive integer; the length of the simulated process.
#' @param p A positive integer; the order of the simulated process.
#' @param alpha A numeric vector whose components belong to the interval \eqn{(0,1)};
#'  parameters of the thinning operator (see details below).
#' @param Y A non-negative integer; the initial value of the process.
#' @param p2 A numeric value between \eqn{0} and \eqn{1}; parameter
#' of the Bernoulli distribution (see details below).
#' @param lamb A positive numeric value; parameter of the Poisson distribution (see details below).
#' @details The ZI-INAR process is defined by the equation 
#' \deqn{Y_t=\sum_{i=1}^p \alpha_i\circ Y_{t-i} +V_t, \qquad t\in \mathbb{Z},} 
#' where the so called thinning operator is defined as \deqn{\alpha\circ Y=\sum_{i=1}^Y Z_i,}
#'  where \eqn{\{Z_i\}_{i\in \mathcal{Z}}} is an i.i.d. sequence of random 
#'  variables with Bernoulli distribution with parameter \eqn{\alpha} with
#'  \eqn{\alpha_i\in[0,1]} for \eqn{i\in 1,\dots,p} and 
#'  independent of \eqn{Y}. Additionally, \eqn{\{V_t\}_{t\in \mathcal{Z}}}
#'   is an i.i.d. non-negative and integer valued sequence of random variables.
#'    
#'    Let \eqn{V} be a non-negative discrete random variable with parameters 
#'    \eqn{p2} and \eqn{\lambda}. We said that \eqn{V} follows a zero-inflated 
#'    distribution and we denote it by \eqn{V\sim ZI(p2,\lambda)}, if it can be expressed as 
#' \deqn{V=BU,\qquad \text{with }\qquad B \perp U,}
#' where \eqn{B} has Bernoulli distribution with mean \eqn{1-p2}, 
#' for \eqn{p2\in[0,1)} 
#' and \eqn{U} is a non-negative discrete valued random
#'  variable with finite variance and parameter vector \eqn{\lambda}. 
#'  We consider the zero-inflated Poisson distribution (ZI-INAR(\eqn{p}))), 
#'  when \eqn{U} has Poisson distribution with mean \eqn{\lambda}.
#' @return  A numeric vector of length \code{n}.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Bertail, P., Medina-Garay, A., De Lima-Medina, F. and Jales, I. (2024). A
#' maximum likelihood and regenerative bootstrap approach for estimation and
#' forecasting of inar (p) processes with zero-inated innovations. \emph{Statistics},
#'  \bold{58}, 336-363.
#'  
#' @seealso \code{\link{regenboot}}, \code{\link{genETARCH}}, \code{\link{genMM1}}.
#' @keywords "ZI-INAR process generation" "Integer autoregressive process" "Zero-inflated"
#' @export
#' @examples 
#' X = zi_inar_process(70, 2, c(0.1,0.1), Y = 1, p2 = 0.2, lamb = 0.7)
#' plot(X, type="l")
##
zi_inar_process <- function(n, p, alpha, Y = 1, p2, lamb){
  if ( (p%%1!=0) || (p<0)) {
    stop("Parameter 'p' must be a natural, possitive number.")}
  if ( sum(alpha <0)+ sum(alpha>1)>0) {
    stop("Parameter 'alpha' must be a number between 0 and 1.")}
  if ( (Y%%1!=0) || (Y<0)) {
    stop("Parameter 'Y' must be a natural, positive number.")}
  if ( (p2 <0) || (p2>=1)) {
    stop("Parameter 'p2' must be a number between 0 and 1.")}
  if ( (lamb<0) ) {
    stop("Parameter 'lamb' must be possitive.")}
  U = rpois(n = n, lambda = lamb)
  B = rbinom(n, 1, 1-p2)
  V = U*B
  Z = rep(0,n)
  for (i in seq(1:n)){
    temp=0
    for (idx in seq(1:p)){
      thin_oper <- sum(rbinom(idx, 1, alpha[idx])) 
      temp = temp + thin_oper
    }
    Z[i]=temp}  
  res <- V + Z
  return(res)}

