bracer
======

[![CRAN Status Badge](https://www.r-pkg.org/badges/version/bracer)](https://cran.r-project.org/package=bracer)
[![R-CMD-check](https://github.com/trevorld/bracer/workflows/R-CMD-check/badge.svg)](https://github.com/trevorld/bracer/actions)
[![Coverage Status](https://codecov.io/github/trevorld/bracer/branch/master/graph/badge.svg)](https://app.codecov.io/github/trevorld/bracer?branch=master)

### Table of Contents

* [Overview](#overview)
* [Examples](#examples)
* [Installation](#installation)
* [Limitations of pure R parser and alternative javascript parser](#limitations)

## <a name="overview">Overview</a>

``bracer`` provides support for performing [brace expansions](https://www.gnu.org/savannah-checkouts/gnu/bash/manual/bash.html#Brace-Expansion) on strings in R.

## <a name="examples">Examples</a>

```r
library("bracer")
options(bracer.engine = "r")
expand_braces("Foo{A..F}")
```

```
## [1] "FooA" "FooB" "FooC" "FooD" "FooE" "FooF"
```

```r
expand_braces("Foo{01..10}")
```

```
##  [1] "Foo01" "Foo02" "Foo03" "Foo04" "Foo05" "Foo06" "Foo07" "Foo08" "Foo09"
## [10] "Foo10"
```

```r
expand_braces("Foo{A..E..2}{1..5..2}")
```

```
## [1] "FooA1" "FooA3" "FooA5" "FooC1" "FooC3" "FooC5" "FooE1" "FooE3" "FooE5"
```

```r
expand_braces("Foo{-01..1}")
```

```
## [1] "Foo-01" "Foo000" "Foo001"
```

```r
expand_braces("Foo{{d..d},{bar,biz}}.{py,bash}")
```

```
## [1] "Food.py"     "Food.bash"   "Foobar.py"   "Foobar.bash" "Foobiz.py"  
## [6] "Foobiz.bash"
```

``expand_braces`` is vectorized and returns one big character vector of all the brace expansions.  ``str_expand_braces`` is an alternative that returns a list of character vectors.


```r
expand_braces(c("Foo{A..F}", "Bar.{py,bash}", "{{Biz}}"))
```

```
## [1] "FooA"     "FooB"     "FooC"     "FooD"     "FooE"     "FooF"     "Bar.py"  
## [8] "Bar.bash" "{{Biz}}"
```

```r
str_expand_braces(c("Foo{A..F}", "Bar.{py,bash}", "{{Biz}}"))
```

```
## [[1]]
## [1] "FooA" "FooB" "FooC" "FooD" "FooE" "FooF"
## 
## [[2]]
## [1] "Bar.py"   "Bar.bash"
## 
## [[3]]
## [1] "{{Biz}}"
```

``glob`` is a wrapper around ``Sys.glob`` that uses ``expand_braces`` to support both brace and wildcard expansion on file paths.


```r
glob("R/*.{R,r,S,s}")
```

```
## [1] "R/engine-r.R"      "R/engine-v8.R"     "R/expand_braces.R"
## [4] "R/glob.R"
```

## <a name="installation">Installation</a>

To install the release version on CRAN use the following command in R:


```r
install.packages("bracer")
```

To install the developmental version use the following command in R:


```r
remotes::install_github("trevorld/bracer")
```

Installing the suggested ``V8`` package will enable use of the javascript alternative parser:


```r
install.packages("V8")
```

## <a name="limitations">Limitations of pure R parser and alternative javascript parser</a>

The ``bracer`` pure R parser currently does not properly support the "correct" (Bash-style) brace expansion under several edge conditions such as:

1. Unbalanced braces e.g. ``{{a,d}`` (but you could use an escaped brace instead ``\\{{a,d}``)
2. Using surrounding quotes to escape terms e.g. ``{'a,b','c'}`` (but you could use an escaped comma instead  ``{a\\,b,c}``)
3. Escaped braces within comma-separated lists e.g. ``{a,b\\}c,d}``
4. (Non-escaping) backslashes before braces e.g. ``{a,\\\\{a,b}c}``
5. Sequences from letters to non-letter ASCII characters e.g. ``X{a..#}X``


```r
options(bracer.engine = "r")
expand_braces("{{a,d}")
```

```
## [1] "{{a,d}"
```

```r
expand_braces("{'a,b','c'}")
```

```
## [1] "'a"  "b'"  "'c'"
```

```r
expand_braces("{a,b\\}c,d}")
```

```
## [1] "a,b}c" "d"
```

```r
expand_braces("{a,\\\\{a,b}c}")
```

```
## [1] "ac}"  "{ac}" "bc}"
```

```r
expand_braces("X{a..#}X")
```

```
## [1] "X{a..#}X"
```

However if the 'V8' suggested R package is installed we can instead use an embedded version of the [braces](https://github.com/micromatch/braces) Javascript library which can correctly handle these edge cases.  To do so we need to set the bracer "engine" to "v8".


```r
options(bracer.engine = "v8")
expand_braces("{{a,d}")
```

```
## [1] "{a" "{d"
```

```r
expand_braces("{'a,b','c'}")
```

```
## [1] "a,b" "c"
```

```r
expand_braces("{a,b\\}c,d}")
```

```
## [1] "a"   "b}c" "d"
```

```r
expand_braces("{a,\\\\{a,b}c}")
```

```
## [1] "a"    "\\ac" "\\bc"
```

```r
expand_braces("X{a..#}X")
```

```
##  [1] "XaX"  "X`X"  "X_X"  "X^X"  "X]X"  "X\\X" "X[X"  "XZX"  "XYX"  "XXX" 
## [11] "XWX"  "XVX"  "XUX"  "XTX"  "XSX"  "XRX"  "XQX"  "XPX"  "XOX"  "XNX" 
## [21] "XMX"  "XLX"  "XKX"  "XJX"  "XIX"  "XHX"  "XGX"  "XFX"  "XEX"  "XDX" 
## [31] "XCX"  "XBX"  "XAX"  "X@X"  "X?X"  "X>X"  "X=X"  "X<X"  "X;X"  "X:X" 
## [41] "X9X"  "X8X"  "X7X"  "X6X"  "X5X"  "X4X"  "X3X"  "X2X"  "X1X"  "X0X" 
## [51] "X/X"  "X.X"  "X-X"  "X,X"  "X+X"  "X*X"  "X)X"  "X(X"  "X'X"  "X&X" 
## [61] "X%X"  "X$X"  "X#X"
```
