% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/operators.R
\name{gradient}
\alias{gradient}
\alias{\%gradient\%}
\title{Numerical and Symbolic Gradient}
\usage{
gradient(
  f,
  var,
  params = list(),
  coordinates = "cartesian",
  accuracy = 4,
  stepsize = NULL,
  drop = TRUE
)

f \%gradient\% var
}
\arguments{
\item{f}{array of \code{characters} or a \code{function} returning a \code{numeric} array.}

\item{var}{vector giving the variable names with respect to which the derivatives are to be computed and/or the point where the derivatives are to be evaluated. See \code{\link{derivative}}.}

\item{params}{\code{list} of additional parameters passed to \code{f}.}

\item{coordinates}{coordinate system to use. One of: \code{cartesian}, \code{polar}, \code{spherical}, \code{cylindrical}, \code{parabolic}, \code{parabolic-cylindrical} or a vector of scale factors for each varibale.}

\item{accuracy}{degree of accuracy for numerical derivatives.}

\item{stepsize}{finite differences stepsize for numerical derivatives. It is based on the precision of the machine by default.}

\item{drop}{if \code{TRUE}, return the gradient as a vector and not as an \code{array} for scalar-valued functions.}
}
\value{
Gradient vector for scalar-valued functions when \code{drop=TRUE}, \code{array} otherwise.
}
\description{
Computes the numerical gradient of \code{functions} or the symbolic gradient of \code{characters} 
in arbitrary \href{https://en.wikipedia.org/wiki/Orthogonal_coordinates#Table_of_orthogonal_coordinates}{orthogonal coordinate systems}.
}
\details{
The gradient of a scalar-valued function \eqn{F} is the vector 
\eqn{(\nabla F)_i} whose components are the partial derivatives of \eqn{F} 
with respect to each variable \eqn{i}. 
The \code{gradient} is computed in arbitrary orthogonal coordinate systems using the 
scale factors \eqn{h_i}:

\deqn{(\nabla F)_i = \frac{1}{h_i}\partial_iF}

When the function \eqn{F} is a tensor-valued function \eqn{F_{d_1,\dots,d_n}}, 
the \code{gradient} is computed for each scalar component. In particular, it becomes
the Jacobian matrix for vector-valued function.

\deqn{(\nabla F_{d_1,\dots,d_n})_i = \frac{1}{h_i}\partial_iF_{d_1,\dots,d_n}}
}
\section{Functions}{
\itemize{
\item \code{f \%gradient\% var}: binary operator with default parameters.

}}
\examples{
### symbolic gradient 
gradient("x*y*z", var = c("x", "y", "z"))

### numerical gradient in (x=1, y=2, z=3)
f <- function(x, y, z) x*y*z
gradient(f = f, var = c(x=1, y=2, z=3))

### vectorized interface
f <- function(x) x[1]*x[2]*x[3]
gradient(f = f, var = c(1, 2, 3))

### symbolic vector-valued functions
f <- c("y*sin(x)", "x*cos(y)")
gradient(f = f, var = c("x","y"))

### numerical vector-valued functions
f <- function(x) c(sum(x), prod(x))
gradient(f = f, var = c(0,0,0))

### binary operator
"x*y^2" \%gradient\% c(x=1, y=3)

}
\references{
Guidotti E (2022). "calculus: High-Dimensional Numerical and Symbolic Calculus in R." Journal of Statistical Software, 104(5), 1-37. \doi{10.18637/jss.v104.i05}
}
\seealso{
Other differential operators: 
\code{\link{curl}()},
\code{\link{derivative}()},
\code{\link{divergence}()},
\code{\link{hessian}()},
\code{\link{jacobian}()},
\code{\link{laplacian}()}
}
\concept{differential operators}
