% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ODEs.R
\name{RK4ODE}
\alias{RK4ODE}
\title{Runge-Kutta 4th order method for systems of ODEs}
\usage{
RK4ODE(f, t0, tf, y0, h, ...)
}
\arguments{
\item{f}{A function of the form \code{f(t,y)} returning a numeric vector.
It must be defined before using \code{RK4ODE}. This function is
the right hand side of the ODE, i.e. the gradient of the ODE system.}

\item{t0}{Initial time.}

\item{tf}{Final time.}

\item{y0}{A numeric vector with initial values (length = \eqn{m}).}

\item{h}{Step size.}

\item{...}{Other parameters potentially needed by the gradient
function.}
}
\value{
A list with elements \code{t} (time points) and \code{y} 
        (solution matrix). The first row of the matrix contains the
        initial values of \code{y} at time \code{t0}. Each column of the
        matrix contains the numerical solution for each one of the \eqn{m}
        functions of the system of ODEs.
}
\description{
Solves a system of \eqn{m} first-order ODEs using the classical 
fourth-order Runge-Kutta method.
}
\details{
This method achieves high accuracy by evaluating the gradient 
function four times per step. It has local error \eqn{O(h^5)} 
and global error \eqn{O(h^4)}. It is one of the most widely used 
methods for solving initial value problems numerically.
}
\examples{

# IVP: \eqn{dy/dt=6-2y,\ y(0)=0}.
# Define gradient
f <- function(t,y) {dy <- 6-2*y; return(dy)}

# Solution interval
t0 <- 0
tf <- 2

# Initial condition
y0 <- 0

# Step
h <- 0.1

# Numerical solution
ltmp <- RK4ODE(f,t0,tf,y0,h)

# Print grid
print(ltmp$t)

# Print numerical solution
print(ltmp$y)

# Example with two ODEs. 
# \eqn{dy_1/dt=y_1+2y_2}
# \eqn{dy_2/dt=(3/2)y_1-y_2}
# \eqn{y_1(0)=1, y_2(0)=-2}

# Define gradient
dy <- function(t,y) {
  dy1 <- y[1]+2*y[2] 
  dy2 <- 1.5*y[1]-y[2] 
  return(c(dy1,dy2))
}

# Solution interval
t0 <- 0
tf <- 2

# Initial conditions
y0 <- c(1,-2)

# Step
h <- 0.1

# Numerical solution
ltmp <- RK4ODE(dy,t0,tf,y0,h)

# Print grid
print(ltmp$t)

# Print numerical solution y1
print(ltmp$y[,1])

# Print numerical solution y2
print(ltmp$y[,2])

}
