% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mantel.R
\name{mantel.test}
\alias{mantel.test}
\alias{mantel.test.default}
\alias{mantel.test.formula}
\alias{mantel.test.list}
\alias{plot.mantel}
\title{Perform one or more Mantel permutation tests.}
\usage{
mantel.test(x, y, ...)

\method{mantel.test}{default}(x, y, plot = FALSE, method = c("spearman",
  "kendall", "pearson"), trials = 9999, omitzerodistances = FALSE, ...)

\method{mantel.test}{formula}(x, y, groups = NULL,
  stringdistfun = utils::adist, meaningdistfun = hammingdists, ...)

\method{mantel.test}{list}(x, y, plot = FALSE, ...)

\method{plot}{mantel}(x, xlab = "generation", ...)
}
\arguments{
\item{x}{a formula, distance matrix, or list of distance matrices (see below)}

\item{y}{a data frame, distance matrix, or list of distance matrices of the
same length as \code{x}}

\item{...}{further arguments which are passed on to the default method (in
particular \code{plot}, \code{method}, \code{trials} and \code{omitzerodistances})}

\item{plot}{logical: immediately produce a plot of the test results (default:
\code{FALSE})}

\item{method}{correlation coefficient to be computed. Passed on to
\code{\link[stats]{cor}}, so one of \code{"spearman"}, \code{"kendall"}, or, inadvisable
in the case of ties: \code{"pearson"}. Following Dietz (1983), \code{"spearman"} is
used as a default that is both powerful and robust across different
distance measures.}

\item{trials}{integer: maximum number of random permutations to be computed
(see Details).}

\item{omitzerodistances}{logical: if \code{TRUE}, the calculation of the
correlation coefficient omits pairs of off-diagonal cells which contain a
0 in the \emph{second} distance matrix argument. (For the formula
interface, this is the matrix which specifies the meaning distances.)}

\item{groups}{when \code{x} is a formula: column name by which the data in \code{y} is
split into separate data sets to run several Mantel tests on}

\item{stringdistfun}{when \code{x} is a formula: edit distance function used to
compute the distance matrix from the specified string column. Supports any
edit distance function that returns a distance matrix from a vector or
list of character strings. Default is Levenshtein distance
(\code{\link[utils]{adist}}), other options from this package include
\code{\link[=normalisedlevenshteindists]{normalisedlevenshteindists()}} and \code{\link[=orderinsensitivedists]{orderinsensitivedists()}}.}

\item{meaningdistfun}{when \code{x} is a formula: meaning distance function used
to compute the distance matrix from the specified meaning columns.
Defaults to Hamming distances between meanings (\code{\link[=hammingdists]{hammingdists()}}), custom
meaning functions can be created easily using
\code{\link[=wrap.meaningdistfunction]{wrap.meaningdistfunction()}}.}

\item{xlab}{the x axis label used when plotting the result of several Mantel
tests next to each other}
}
\value{
A dataframe of class \code{mantel}, with one row per Mantel test carried
out, containing the following columns:
\describe{
\item{\code{method}}{Character string: type of correlation coefficient used}
\item{\code{statistic}}{The veridical correlation coefficient between
the entries in the two distance matrices}
\item{\code{rsample}}{A list of correlation coefficients calculated
from the permutations of the input matrices}
\item{\code{mean}}{Average correlation coefficient produced by the permutations}
\item{\code{sd}}{Standard deviation of the sampled correlation coefficients}
\item{\code{p.value}}{Empirical p-value computed from the Mantel
test: let \code{ngreater} be the number of correlation coefficients
in \code{rsample} greater than or equal to \code{statistic}, then
\code{p.value} is \code{(ngreater+1)/(length(rsample)+1}}
\item{\code{p.approx}}{The theoretical p-value that would correspond
to the standard \code{z} score as calculated above.}
\item{\code{is.unique.max}}{Logical, \code{TRUE} iff the veridical
correlation coefficient is greater than any of the coefficients
calculated for the permutations. If this is true, then
\code{p.value == 1 / (length(rsample)+1)}}
}
Multiple \code{mantel} objects can easily be combined by calling
\code{rbind(test1, test2, ...)}.
}
\description{
Perform correlation tests between pairs of distance matrices. The Mantel
test is different from classical correlation tests (such as those
implemented by \code{\link[stats]{cor.test}}) in that the null distribution
(and significance level) are obtained through randomisation. The null
distribution is generated by shuffling the locations (matrix rows and
columns) of one of the matrices to calculate an empirical null distribution
for the given data set.
}
\details{
If the number of possible permutations of the matrices is reasonably close
to the number of permutations specified by the \code{trials} parameter, a
deterministic enumeration of all the permutations will be carried out
instead of random sampling: such a deterministic test will return an exact
p-value.

\code{plot()} called on a data frame of class \code{mantel} plots a
visualisation of the test results (in particular, the distribution of
the permutated samples against the veridical correlation coefficient). If
the veridical correlation coefficient is plotted in blue it means
that it was higher than all other coefficients generated by random
permutations of the data. When the argument contains the result of more than
one Mantel tests, a side-by-side boxplot visualisation shows the mean and
standard deviation of the randomised samples (see examples). Additional
parameters \code{...} to \code{plot()} are passed on to
\code{\link[graphics]{plot.default}}.
}
\section{Methods (by class)}{
\itemize{
\item \code{default}: Perform Mantel correlation test on two distance
matrices. The distance matrices can either be of type
\code{\link[stats]{dist}}, plain R matrices or any object that can be
interpreted by \code{\link{check.dist}}. The order of the two matrices does
not matter unless \code{omitzerodistances = TRUE}, in which case cells with
a 0 in the \emph{second} matrix are omitted from the calculation of the
correlation coefficient. For consistency it is therefore recommended to
always pass the string distance matrix first, meaning distance matrix second.

\item \code{formula}: This function can be called with raw experimental
result data frames, distance matrix calculation is taken care of internally.
\code{x} is a formula of the type \code{s ~ m1 + m2 + ...} where \code{s}
is the column name of the character strings in data frame or matrix \code{y},
while \code{m1} etc. are the column names specifying the different meaning
dimensions. To calculate the respective distances, the function
\code{stringdistfun} is applied to the strings, \code{meaningdistfun} to the
meaning columns.

\item \code{list}: When \code{x} is a list of distance matrices, and
\code{y} is either a single distance matrix or a list of distance matrices
the same length as \code{x}: runs a Mantel test for every pairwise
combination of distance matrices in \code{x} and \code{y} and returns a
\code{mantel} object with as many rows.
}}

\examples{
# small distance matrix, Mantel test run deterministically
mantel.test(dist(1:7), dist(1:7))


\dontrun{
# run test on smallest distance matrix which requires a random
# permutation test, and plot it
plot(mantel.test(dist(1:8), dist(1:8), method="kendall"))
}

\dontrun{
# 2x2x2x2 design
mantel.test(hammingdists(enumerate.meaningcombinations(c(2, 2, 2, 2))),
  dist(1:16), plot=TRUE)
}

# using the formula interface in combination with a data frame:
print(data <- cbind(word=c("aa", "ab", "ba", "bb"),
  enumerate.meaningcombinations(c(2, 2))))

mantel.test(word ~ Var1 + Var2, data)

\dontrun{
# pass a list of distance matrices as the first argument, but just one
# distance matrix as the second argument: this runs separate tests on
# the pairwise combinations of the first and second argument
result <- mantel.test(list(dist(1:8), dist(sample(8:1)), dist(runif(8))),
  hammingdists(enumerate.meaningcombinations(c(2, 2, 2))))

# print the result of the three independently run permutation tests
print(result)

# show the three test results in one plot
plot(result, xlab="group")
}
}
\references{
Dietz, E. J. 1983 “Permutation Tests for Association
Between Two Distance Matrices.” \emph{Systematic Biology} 32 (1): 21-–26.
\url{https://doi.org/10.1093/sysbio/32.1.21}.

North, B. V., D. Curtis and P. C. Sham. 2002 “A Note on the Calculation of
Empirical P Values from Monte Carlo Procedures.” \emph{The American Journal of
Human Genetics} 71 (2): 439-–41. \url{https://doi.org/10.1086/341527}.
}
\seealso{
\code{\link[stats]{cor}},
\code{\link[utils]{adist}}, \code{\link{hammingdists}},
\code{\link{normalisedlevenshteindists}},
\code{\link{orderinsensitivedists}}
}
