% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prepare_trait_data.R
\name{BSM_to_phytools_simmap}
\alias{BSM_to_phytools_simmap}
\alias{BSMs_to_phytools_simmaps}
\title{Convert Biogeographic Stochastic Map (BSM) to phytools SIMMAP stochastic map (SM) format}
\usage{
BSM_to_phytools_simmap(model_fit, phylo, BSM_output, sim_index)

BSMs_to_phytools_simmaps(model_fit, phylo, BSM_output)
}
\arguments{
\item{model_fit}{A BioGeoBEARS results object, produced by ML inference via \code{BioGeoBEARS::bears_optim_run()}.}

\item{phylo}{Time-calibrated phylogeny used in the BioGeoBEARS analyses to produce the historical biogeographic inference
and run the Biogeographic Stochastic Mapping. Object of class \code{"phylo"} as defined in \code{{ape}}.}

\item{BSM_output}{A list with two objects, a cladogenetic events table and an anagenetic events table, as the result of
Biogeographic Stochastic Mapping conducted with \code{BioGeoBEARS::runBSM()}.}

\item{sim_index}{Integer. Index of the biogeographic simulation targeted to produce the \code{simmap} with \code{\link[=BSM_to_phytools_simmap]{BSM_to_phytools_simmap()}}.}
}
\value{
The \code{\link[=BSM_to_phytools_simmap]{BSM_to_phytools_simmap()}} function returns a list with two elements:
\itemize{
\item \verb{$simmap} A unique \code{simmap} for a given biogeographic simulation as an object of classes \code{c("simmap", "phylo")}.
This is a modified \code{{ape}} tree with additional elements to report range mapping, model parameters and likelihood.
\itemize{
\item \verb{$maps} A list of named numerical vectors. Provides the mapping of ranges along each remaining edge.
Names are the ranges. Values are residence times in each state across segments
\item \verb{$mapped.edge} A numerical matrix. Provides the evolutionary time spent across ranges (columns) along the edges (rows).
row.names() are the node ID at the rootward and tipward ends of each edge.
\item \verb{$Q} Numerical matrix. The transition rates across ranges calculated from the ML parameter estimates of the model.
\item \verb{$logL} Numeric. The log-likelihood of the data under the ML model.
}
\item \verb{$residence_times} Data.frame with two rows. Summarizes the residence time spent in each range along all branches,
in (raw) evolutionary time (i.e., branch lengths), and in percentage (perc).
}

The \code{\link[=BSMs_to_phytools_simmaps]{BSMs_to_phytools_simmaps()}} function loop around the \code{\link[=BSM_to_phytools_simmap]{BSM_to_phytools_simmap()}} function to aggregate all \code{simmaps}
from all biogeographic simulations in a unique list of classes \code{c("multiSimmap", "multiPhylo")}.
\itemize{
\item Each element in the \verb{$simmap} of a biogeographic simulation obtained with \code{\link[=BSM_to_phytools_simmap]{BSM_to_phytools_simmap()}}.
\item \verb{$residence_times} summary data.frames are not preserved.
}
}
\description{
These functions converts a Biogeographic Stochastic Map (BSM) output from BioGeoBEARS into
a \code{simmap} object from R package \code{{phytools}} (See \code{\link[phytools:make.simmap]{phytools::make.simmap()}}).

They require a model fit with \code{BioGeoBEARS::bears_optim_run()} and the output of a Biogeographic Stochastic Mapping
performed with \code{BioGeoBEARS::runBSM()} to produce \code{simmap} objects as phylogenies with the associated
mapping of range evolution along branches across simulations.
\itemize{
\item \code{\link[=BSM_to_phytools_simmap]{BSM_to_phytools_simmap()}}: Produce one \code{simmap} for the required simulation (index of the simulation provided with \code{sim_index}).
\item \code{\link[=BSMs_to_phytools_simmaps]{BSMs_to_phytools_simmaps()}}: Produce all \code{simmap} objects for all simulations stored in a unique \code{multiSimmap} object.
}

Initial functions in R package BioGeoBEARS by Nicholas J. Matzke:
\itemize{
\item \code{BioGeoBEARS::BSM_to_phytools_SM()}
\item \code{BioGeoBEARS::BSMs_to_phytools_SMs()}
}
}
\details{
These functions are slight adaptations of original functions from the R Package \code{BioGeoBEARS} by N. Matzke.

Initial functions: \code{BioGeoBEARS::BSM_to_phytools_SM()} \code{BioGeoBEARS::BSMs_to_phytools_SMs()}

Changes:
\itemize{
\item Solves issue with differences in ranges allowed across time-strata.
\item Requires directly the output of \code{BioGeoBEARS::runBSM()} instead of separated cladogenetic and anagenetic event tables.
\item Update the documentation.
}
}
\section{Notes on BioGeoBEARS}{
The R package \code{BioGeoBEARS} is needed for this function to work with biogeographic data.
Please install it manually from: \href{https://github.com/nmatzke/BioGeoBEARS}{https://github.com/nmatzke/BioGeoBEARS}.
}

\section{Notes on using the resulting simmap object in phytools (adapted from Nicholas J. Matzke)}{
The phytools functions, like \code{\link[phytools:countSimmap]{phytools::countSimmap()}}, will only count the anagenetic events
(range transitions occurring along branches) as it was written assuming purely anagenetic models.

It remains possible to extract cladogenetic events (range transitions occurring at speciation)
by comparing the last-state-below-a-node with the descendant-pairs-above-a-node.
However, it is recommended to use the built-in functions from BioGeoBEARS to summarize
the biogeographic history based on the tables of cladogenetic and anagenetic events obtained
from \code{BioGeoBEARS::runBSM()}. \code{simmap} objects should primarily be considered as a tool for visualization.

Associated functions in R package \code{BioGeoBEARS}:
\itemize{
\item \code{BioGeoBEARS::simulate_source_areas_ana_clado()}: To select randomly a unique area source for transition from a multi-area state to a single area.
\item \code{BioGeoBEARS::get_dmat_times_from_res()}: To generate matrices of range expansion from source area to destination area.
\item \code{BioGeoBEARS::count_ana_clado_events()}: To count the number and type of events from BSM tables.
\item \code{BioGeoBEARS::hist_event_counts()}: To plot histograms of event counts across BSM tables.
}

Please note carefully that area-to-area dispersal events are not identical with the state transitions.
For example, a state can be a geographic range with multipleareas, but under the logic of DEC-type models,
a range-expansion event like  ABC->ABCD actually means that a dispersal happened from some specific area (A, B, or C)
to the new area. BSMs track this area-to-area sourcing in its cladogenetic and anagenetic event tables,
at least if \code{BioGeoBEARS::simulate_source_areas_ana_clado()} has been run on the output of \code{BioGeoBEARS::runBSM()}.
}

\examples{
if (deepSTRAPP::is_dev_version())
{
 ## Run only if you have R package 'BioGeoBEARS' installed.
 # Please install it manually from: https://github.com/nmatzke/BioGeoBEARS")

 ## Load phylogeny and tip data
 library(phytools)
 data(eel.tree)

 \donttest{ # (May take several minutes to run)
 ## Load directly output of prepare_trait_data() run on biogeographic data
 data(eel_biogeo_data, package = "deepSTRAPP")

 ## Convert BSM output into a unique simmap, including residence times
 simmap_1 <- BSM_to_phytools_simmap(model_fit = eel_biogeo_data$best_model_fit,
                                    phylo = eel.tree,
                                    BSM_output = eel_biogeo_data$BSM_output,
                                    sim_index = 1)
 # Explore output
 str(simmap_1, max.level = 1)
 # Print residence times in each range
 simmap_1$residence_times
 # Plot simmap
 plot(simmap_1$simmap)

 ## Convert BSM output into all simmaps in a multiSimmap/multiPhylo object
 all_simmaps <- BSMs_to_phytools_simmaps(model_fit = eel_biogeo_data$best_model_fit,
                                         phylo = eel.tree,
                                         BSM_output = eel_biogeo_data$BSM_output)
 # Explore output
 str(all_simmaps, max.level = 1)
 # Plot simmap n°1
 plot(all_simmaps[[1]]) }
}

}
\references{
For BioGeoBEARS: Matzke, Nicholas J. (2018). BioGeoBEARS: BioGeography with Bayesian (and likelihood) Evolutionary Analysis with R Scripts.
version 1.1.1, published on GitHub on November 6, 2018. \doi{10.5281/zenodo.1478250}. Website: \url{http://phylo.wikidot.com/biogeobears}.
}
\seealso{
\code{\link[phytools:countSimmap]{phytools::countSimmap()}} \code{\link[phytools:make.simmap]{phytools::make.simmap()}}
\code{BioGeoBEARS::simulate_source_areas_ana_clado()} \code{BioGeoBEARS::get_dmat_times_from_res()}
\code{BioGeoBEARS::count_ana_clado_events()} \code{BioGeoBEARS::hist_event_counts()}
}
\author{
Nicholas J. Matzke. Contact: \email{matzke@berkeley.edu}

Changes by Maël Doré (see Details)
}
