% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/degreestatrem.R
\name{remstats_degree}
\alias{remstats_degree}
\title{Compute Degree Network Statistics for Event Senders and Receivers in a Relational Event Sequence}
\usage{
remstats_degree(
  formation = c("sender-indegree", "receiver-indegree", "sender-outdegree",
    "receiver-outdegree"),
  time,
  sender,
  receiver,
  observed,
  sampled,
  halflife = 2,
  counts = FALSE,
  dyadic_weight = 0,
  exp_weight_form = FALSE
)
}
\arguments{
\item{formation}{The degree statistic to be computed. "sender-indegree" computes the indegree statistic for the event senders. "receiver-indegree" computes the
indegree statistic for the event receivers. "sender-outdegree" computes the outdegree statistic for the event senders. "receiver-outdegree" computes the
outdegree statistic for the event receivers.}

\item{time}{The vector of event times from the post-processing event sequence.}

\item{sender}{The vector of event senders from the post-processing event sequence.}

\item{receiver}{The vector of event receivers from the post-processing event sequence}

\item{observed}{A vector for the post-processing event sequence where i is equal to 1 if the dyadic event is observed and 0 if not.}

\item{sampled}{A vector for the post-processing event sequence where i is equal to 1 if the observed dyadic event is sampled and 0 if not.}

\item{halflife}{A numerical value that is the halflife value to be used in the exponential weighting function (see details section). Preset to 2 (should be updated by the user based on substantive context).}

\item{counts}{TRUE/FALSE. TRUE indicates that the counts of past events should be computed (see the details section). FALSE indicates that the temporal exponential weighting function should be used to downweigh past events (see the details section). Set to FALSE by default.}

\item{dyadic_weight}{A numerical value for the dyadic cutoff weight that represents the numerical cutoff value for temporal relevancy based on the exponential weighting function. For example, a numerical value of 0.01, indicates that an exponential weight less than 0.01 will become 0 and that events with such value (or smaller values) will not be included in the sum of the past event weights (see the details section). Set to 0 by default.}

\item{exp_weight_form}{TRUE/FALSE. TRUE indicates that the Lerner et al. (2013) exponential weighting function will be used (see the details section). FALSE indicates that the Lerner and Lomi (2020) exponential weighting function will be used (see the details section). Set to FALSE by default}
}
\value{
The vector of degree statistics for the relational event sequence.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

The function computes the indegree network sufficient statistic for event senders
in a relational event sequence (see Lerner and Lomi 2020; Butts 2008).
This measure allows for indegree scores to be only  computed for the sampled
events, while creating the weights based on the full event sequence (see
Lerner and Lomi 2020; Vu et al. 2015). The function also allows users to use two
different weighting functions, return the counts of past events, reduce computational
runtime, and specify a dyadic cutoff for relational relevancy.
}
\details{
The function calculates sender indegree scores for relational event
sequences based on the exponential weighting function used in either Lerner
and Lomi (2020) or Lerner et al. (2013).

Following Lerner and Lomi (2020), the exponential weighting function in
relational event models is:
\deqn{w(s, r, t) = e^{-(t-t') \cdot \frac{ln(2)}{T_{1/2}} }}

Following Lerner et al. (2013), the exponential weighting function in
relational event models is:
\deqn{w(s, r, t) = e^{-(t-t') \cdot \frac{ln(2)}{T_{1/2}} } \cdot \frac{ln(2)}{T_{1/2}}}

In both of the above equations, \emph{s} is the current event sender, \emph{r} is the
current event receiver (target), \emph{t} is the current event time, \emph{t'} is the
past event times that meet the weight subset, and \eqn{T_{1/2}} is the halflife parameter.

\strong{Sender-Indegree Statistic}:

The formula for sender indegree for event \eqn{e_i} is:
\deqn{sender indegree_{e_{i}} = w(s', s, t) }

That is, all past events in which the event receiver is the current sender. Following Butts (2008), if the
counts of the past events are requested, the formula for sender indegree for
event \eqn{e_i} is:
\deqn{sender indegree_{e_{i}} = d(r' = s, t') }
Where, \eqn{d()} is the number of past events where the event receiver, \emph{r'}, is the current event sender \emph{s} .

\strong{Sender-Outdegree Statistic}:

The formula for sender outdegree for event \eqn{e_i} is:
\deqn{sender outdegree_{e_{i}} = w(s, r', t) }

That is, all past events in which the past sender is the current sender and
the event target can be any past user. Following Butts (2008), if the counts
of the past events are requested, the formula for sender outdegree for
event \eqn{e_i} is:
\deqn{sender outdegree_{e_{i}} = d(s = s', t') }
Where, \eqn{d()} is the number of past events where the sender \emph{s'} is the current event sender, \emph{s}

\strong{Receiver-Outdegree Statistic}:

The formula for receiver outdegree for event \eqn{e_i} is:
\deqn{receiver outdegree_{e_{i}} = w(r', r, t) }

Following Butts (2008), if the counts of the past events are requested, the formula for receiver outdegree for
event \eqn{e_i} is:
\deqn{receiver outdegree{e_{i}} = d(s' = r, t') }
Where, \eqn{d()} is the number of past events where the event sender, \emph{s'}, is the current event receiver, \emph{r'}.

\strong{Receiver-Indegree Statistic}:

The formula for receiver indegree for event \eqn{e_i} is:
\deqn{reciever indegree_{e_{i}} = w(s', r, t) }

That is, all past events in which the event receiver is the current receiver.
Following Butts (2008), if the counts of the past events are requested, the formula for receiver indegree for
event \eqn{e_i} is:
\deqn{reciever indegree_{e_{i}} = d(r' = r, t') }
where, \eqn{d()} is the number of past events where the past event receiver, \emph{r'}, is the
current event receiver (target).

Lastly, researchers interested in modeling temporal relevancy (see Quintane,
Mood, Dunn, and Falzone 2022; Lerner and Lomi 2020) can specify the dyadic
weight cutoff, that is, the minimum value for which the weight is considered
relationally relevant. Users who do not know the specific dyadic cutoff value to use, can use the
\code{\link{remstats_dyadcut}} function.
}
\examples{
events <- data.frame(time = 1:18, eventID = 1:18,
                                sender = c("A", "B", "C",
                                           "A", "D", "E",
                                           "F", "B", "A",
                                           "F", "D", "B",
                                           "G", "B", "D",
                                          "H", "A", "D"),
                               target = c("B", "C", "D",
                                          "E", "A", "F",
                                          "D", "A", "C",
                                          "G", "B", "C",
                                          "H", "J", "A",
                                          "F", "C", "B"))

eventSet <- create_riskset(type = "one-mode",
                      time = events$time,
                      eventID = events$eventID,
                      sender = events$sender,
                      receiver = events$target,
                      p_samplingobserved = 1.00,
                      n_controls = 1,
                      seed = 9999)

#Computing the sender indegree statistic for the relational event sequence
eventSet$senderind <- remstats_degree(
   formation = "sender-indegree",
   time = as.numeric(eventSet$time),
   observed = eventSet$observed,
   sampled = rep(1,nrow(eventSet)),
   sender = eventSet$sender,
   receiver = eventSet$receiver,
   halflife = 2, #halflife parameter
   dyadic_weight = 0,
   exp_weight_form = FALSE)

#Computing the sender outdegree statistic for the relational event sequence
eventSet$senderout <- remstats_degree(
   formation = "sender-outdegree",
   time = as.numeric(eventSet$time),
   observed = eventSet$observed,
   sampled = rep(1,nrow(eventSet)),
   sender = eventSet$sender,
   receiver = eventSet$receiver,
   halflife = 2, #halflife parameter
   dyadic_weight = 0,
   exp_weight_form = FALSE)


#Computing the receiver outdegree statistic for the relational event sequence
eventSet$recieverout <- remstats_degree(
   formation = "receiver-outdegree",
   time = as.numeric(eventSet$time),
   observed = eventSet$observed,
   sampled = rep(1,nrow(eventSet)),
   sender = eventSet$sender,
   receiver = eventSet$receiver,
   halflife = 2, #halflife parameter
   dyadic_weight = 0,
   exp_weight_form = FALSE)


#Computing the receiver indegree statistic for the relational event sequence
eventSet$recieverind <- remstats_degree(
   formation = "receiver-indegree",
   time = as.numeric(eventSet$time),
   observed = eventSet$observed,
   sampled = rep(1,nrow(eventSet)),
   sender = eventSet$sender,
   receiver = eventSet$receiver,
   halflife = 2, #halflife parameter
   dyadic_weight = 0,
   exp_weight_form = FALSE)
}
\references{
Butts, Carter T. 2008. "A Relational Event Framework for Social Action." \emph{Sociological Methodology} 38(1): 155-200.

Quintane, Eric, Martin Wood, John Dunn, and Lucia Falzon. 2022. “Temporal
Brokering: A Measure of Brokerage as a Behavioral Process.” \emph{Organizational Research Methods}
25(3): 459-489.

Lerner, Jürgen and Alessandro Lomi. 2020. “Reliability of relational event
model estimates under sampling: How to fit a relational event model to 360
million dyadic events.” \emph{Network Science} 8(1): 97-135.

Lerner, Jürgen, Margit Bussman, Tom A.B. Snijders, and Ulrik Brandes. 2013. "
Modeling Frequency and Type of Interaction in Event Networks."
\emph{The Corvinus Journal of Sociology and Social Policy} 4(1): 3-32.

Vu, Duy, Philippa Pattison, and Garry Robins. 2015. "Relational event models for social learning in MOOCs." \emph{Social Networks} 43: 121-135.
}
\author{
Kevin A. Carson \href{mailto:kacarson@arizona.edu}{kacarson@arizona.edu}, Diego F. Leal \href{mailto:dflc@arizona.edu}{dflc@arizona.edu}
}
