% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ddbs_read.R
\name{ddbs_read_vector}
\alias{ddbs_read_vector}
\title{Load spatial vector data from DuckDB into R}
\usage{
ddbs_read_vector(
  conn,
  name,
  crs = NULL,
  crs_column = "crs_duckspatial",
  clauses = NULL,
  quiet = FALSE
)
}
\arguments{
\item{conn}{A connection object to a DuckDB database}

\item{name}{A character string of length one specifying the name of the table,
or a character string of length two specifying the schema and table
names. If \code{NULL} (the default), the function returns the result as an
\code{sf} object}

\item{crs}{The coordinates reference system of the data. Specify if the data
doesn't have a \code{crs_column}, and you know the CRS.}

\item{crs_column}{a character string of length one specifying the column
storing the CRS (created automatically by \code{\link{ddbs_write_vector}}).
Set to \code{NULL} if absent.}

\item{clauses}{character, additional SQL code to modify the query from the
table (e.g. "WHERE ...", "ORDER BY...")}

\item{quiet}{A logical value. If \code{TRUE}, suppresses any informational messages.
Defaults to \code{FALSE}.}
}
\value{
an sf object
}
\description{
Retrieves the data from a DuckDB table, view, or Arrow view with a geometry
column, and converts it to an R \code{sf} object. This function works with
both persistent tables created by \code{ddbs_write_vector} and temporary
Arrow views created by \code{ddbs_register_vector}.
}
\examples{
\dontshow{if (interactive()) withAutoprint(\{ # examplesIf}
## load packages
library(duckspatial)
library(sf)

# create a duckdb database in memory (with spatial extension)
conn <- ddbs_create_conn(dbdir = "memory")

## create random points
random_points <- data.frame(
  id = 1:5,
  x = runif(5, min = -180, max = 180),
  y = runif(5, min = -90, max = 90)
)

## convert to sf
sf_points <- st_as_sf(random_points, coords = c("x", "y"), crs = 4326)

## Example 1: Write and read persistent table
ddbs_write_vector(conn, sf_points, "points")
ddbs_read_vector(conn, "points", crs = 4326)

## Example 2: Register and read Arrow view (faster, temporary)
ddbs_register_vector(conn, sf_points, "points_view")
ddbs_read_vector(conn, "points_view", crs = 4326)

## disconnect from db
ddbs_stop_conn(conn)
\dontshow{\}) # examplesIf}
}
